/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/fe/AlarmTransitionBitsEditor
 */
define(['baja!',
  'baja!alarm:AlarmTransitionBits',
  'bajaux/events',
  'jquery',
  'Promise',
  'underscore',
  'nmodule/alarm/rc/baja/AlarmTransitionBits',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'css!nmodule/alarm/rc/fe/alarmEditors'], function (baja,
                                                     types,
                                                     events,
                                                     $,
                                                     Promise,
                                                     _,
                                                     AlarmTransitionBits,
                                                     fe,
                                                     BaseEditor) {

  'use strict';

  var SHOW_KEYS        = [
        AlarmTransitionBits.SHOW_OFF_NORMAL,
        AlarmTransitionBits.SHOW_FAULT,
        AlarmTransitionBits.SHOW_NORMAL,
        AlarmTransitionBits.SHOW_ALERT
      ],
      DISPLAY_KEYS     = [
        'toOffnormal',
        'toFault',
        'toNormal',
        'toAlert'
      ],

      DESTROY_EVENT    = events.DESTROY_EVENT,
      DISABLE_EVENT    = events.DISABLE_EVENT,
      ENABLE_EVENT     = events.ENABLE_EVENT,
      INITIALIZE_EVENT = events.INITIALIZE_EVENT,
      LOAD_EVENT       = events.LOAD_EVENT,
      MODIFY_EVENT     = events.MODIFY_EVENT,
      READONLY_EVENT   = events.READONLY_EVENT,
      WRITABLE_EVENT   = events.WRITABLE_EVENT;


  /**
   * Editor for working with `alarm:AlarmTransitionBits` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/alarm/rc/fe/AlarmTransitionBitsEditor
   */
  var AlarmTransitionBitsEditor = function AlarmTransitionBitsEditor() {
    BaseEditor.apply(this, arguments);
  };
  AlarmTransitionBitsEditor.prototype = Object.create(BaseEditor.prototype);
  AlarmTransitionBitsEditor.prototype.constructor = AlarmTransitionBitsEditor;

  /**
   * Initialize boolean editors for the four available bits of an
   * `AlarmTransitionBits`. If any Properties for a particular bit (e.g.
   * `AlarmTransitionBits.SHOW_ALERT`) are set to `false`, those editors will
   * be hidden.
   *
   * @param {JQuery} dom
   */
  AlarmTransitionBitsEditor.prototype.doInitialize = function (dom) {
    var that  = this,
        props = that.properties();

    dom.on([
      DESTROY_EVENT,
      DISABLE_EVENT, ENABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT,
      READONLY_EVENT, WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    return Promise.all(_.map(SHOW_KEYS, function (showKey, i) {
      var text   = DISPLAY_KEYS[i],
          subDom = $('<div/>').addClass(showKey).appendTo(dom);

      if (props.getValue(showKey) === false) {
        subDom.hide();
      }

      return fe.buildFor({
        dom: subDom,
        properties: { trueText: text, falseText: text },
        value: false,
        formFactor: 'mini'
      });
    }));
  };

  /**
   * Set the values of the four boolean editors according to whether the
   * `AlarmTransitionBits` has those bits set.
   *
   * @param {module:nmodule/alarm/rc/baja/AlarmTransitionBits} transitionBits an `alarm:AlarmTransitionBits` value
   */
  AlarmTransitionBitsEditor.prototype.doLoad = function (transitionBits) {
    var kids = this.getChildEditors(),
        bits = transitionBits.$bits;

    return Promise.all(_.map(SHOW_KEYS, function (showKey, i) {
      return kids[i].load(!!(bits & (1 << i)));
    }));
  };

  /**
   * Resolve a new `alarm:AlarmTransitionBits` instance with the bits set
   * according to the current values of the boolean editors.
   *
   * @returns {Promise}
   */
  AlarmTransitionBitsEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll().spread(AlarmTransitionBits.make);
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  AlarmTransitionBitsEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  AlarmTransitionBitsEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  AlarmTransitionBitsEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };


  return AlarmTransitionBitsEditor;
});
