/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/editors/RunCommand
 */
define(['baja!',
  'jquery',
  'Promise',
  'nmodule/webEditors/rc/fe/fe',
  'lex!alarm',
  'dialogs',
  'nmodule/alarm/rc/Alarm',
  'nmodule/alarm/rc/console/baja/BajaDataChannel',
  'bajaux/commands/Command',
  'bajaux/util/CommandButton',
  'hbs!nmodule/alarm/rc/db/templates/dialog',
  'css!nmodule/alarm/rc/db/templates/view'
], function (baja, $, Promise, fe, lexs, dialogs, Alarm, BajaDataChannel, Command, CommandButton, dialogTemplate) {

  "use strict";

  var lex                             = lexs[0],
      WARNING_ICON                    = lex.getSafe('alarm.db.warningIcon'),
      RUN_MAINTENANCE_BUTTON_SELECTOR = '.run',
      ALARM_TABLE_SELECTOR            = '.alarmTable',
      RUN_CMD_LABEL                   = lex.get('command.runMaintenance.label'),
      CONFIRM_CLEAR_MSG               = lex.get('confirmClear.title'),
      COMMAND_SUCCESS_MSG             = lex.getSafe('alarm.db.successMessage'),
      COMMAND_ERROR_MSG               = lex.getSafe('alarm.db.errorMessage'),
      INVALID_TIME_TITLE              = lex.get('alarm.db.invalidTime'),
      NO_RECORD_SELECTED              = lex.getSafe('alarm.noRecordSelected'),
      CANNOT_BE_UNDONE                = lex.get('cannotBeUndone'),
      CONTINUE                        = lex.get('continue'),
      DELAY                           = 1500,
      warningIcon                     = baja.Icon.make(WARNING_ICON).getImageUris()[0],
      operations                      = {
        'clearOld': {op: Alarm.clearOldRecords, arg: null},
        'clearSelected': {op: Alarm.clearRecords, arg: null},
        'clearAllBeforeSelected': {op: Alarm.clearOldRecords, arg: null},
        'clearAll': {op: Alarm.clearAllRecords, arg: null}
      };

  var RunCommand = function () {
  };

  RunCommand.makeFor = function (view) {
    return fe.buildFor({
      dom: $(RUN_MAINTENANCE_BUTTON_SELECTOR, view.jq()),
      type: CommandButton,
      value: new Command({
        displayName: RUN_CMD_LABEL,
        func: function () {
          return new Promise(function () {
            /*jshint -W030 */
            var key          = $("input[type='radio']:checked", view.jq()).val(),
                operation    = operations[key],
                uuids,
                table        = $(ALARM_TABLE_SELECTOR, view.jq()).data('widget'),
                model        = table.getModel(),
                selectedRows = table.$getSelection().getSelectedElements(model.getRows()),
                dialogContent;

            function commitOperation() {
              var args = [];

              //push the function arg if the current op has one
              if (operation.arg) {
                args.push(operation.arg);
              }

              //push the callback object
              args.push({
                ok: function () {
                  dialogs.showOk({
                    content: COMMAND_SUCCESS_MSG,
                    ok: function () {
                      //NCCB-16175: Clear any selected rows
                      table.$getSelection().clear();

                      //update the table
                      if (view.pager) {
                        view.pager.update();
                      }
                    }
                  });
                },

                fail: function () {
                  dialogs.showOk({content: COMMAND_ERROR_MSG});
                }
              });

              dialogs.showYesNo({
                title: CONFIRM_CLEAR_MSG,
                content: dialogContent,
                yes: function () {
                  dialogs.showLoading(DELAY, operation.op.apply(null, args));
                }
              });
            }

            function clearOldRecords() {
              if (!view.dateEditor) {
                return;
              }

              view.dateEditor.read().then(function (beforeTime) {
                switch (key) {
                  case "clearOld":
                    operation.arg = beforeTime.encodeToString();
                    dialogContent = dialogTemplate({
                      icon: warningIcon,
                      message: lex.get({ key: 'alarm.db.clearBeforeTimestamp', args: [BajaDataChannel.createDateString(beforeTime.getJsDate())] }),
                      cannotBeUndone: CANNOT_BE_UNDONE,
                      continue: CONTINUE
                    });
                    break;

                  case "clearAllBeforeSelected":
                    if (table) {
                      if (selectedRows.length) {

                        if (selectedRows.length > 1) {
                          dialogs.showOk({
                            content: lex.getSafe('alarm.multipleRecords')
                          });
                          return;
                        }

                        else {
                          operation.arg = selectedRows[0].getSubject().get('timestamp').encodeToString();
                        }
                      }

                      else {
                        dialogs.showOk({content: NO_RECORD_SELECTED});
                        return;
                      }

                      dialogContent = dialogTemplate({
                        icon: warningIcon,
                        message: lex.get('alarm.db.clearOlderAlarms'),
                        cannotBeUndone: CANNOT_BE_UNDONE,
                        continue: CONTINUE
                      });
                    }
                    break;

                  default:
                    operation.arg = null;
                }

                commitOperation();
              }).catch(function (err) {
                dialogs.showOk({
                  title: INVALID_TIME_TITLE,
                  content: err
                });
              });
            }

            function clearSelectedRecords() {
              uuids = [];

              if (selectedRows.length) {
                selectedRows.forEach(function (row) {
                  uuids.push(row.getSubject().get('uuid').encodeToString());
                });
              }

              else {
                dialogs.showOk({content: NO_RECORD_SELECTED});
                return;
              }

              operation.arg = uuids;

              dialogContent = dialogTemplate({
                icon: warningIcon,
                message: lex.get('alarm.db.clearSelectedAlarms'),
                cannotBeUndone: CANNOT_BE_UNDONE,
                continue: CONTINUE
              });

              commitOperation();
            }

            function clearAllRecords() {

              if (!model.getRows().length) {
                dialogs.showOk(lex.getSafe('alarm.db.isEmpty'));
                return;
              }

              //clear all alarms
              dialogContent = dialogTemplate({
                icon: warningIcon,
                message: lex.get('alarm.db.clearAllAlarms'),
                cannotBeUndone: CANNOT_BE_UNDONE,
                continue: CONTINUE
              });
              commitOperation();
            }

            if (operation.op === Alarm.clearOldRecords) {
              clearOldRecords();
            }

            else if (operation.op === Alarm.clearRecords) {
              clearSelectedRecords();
            }

            else {
              clearAllRecords();
            }
          });
        }
      })
    });
  };

  return RunCommand;
});

