/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/commands/NotesCommand
 */
define([ 'baja!',
        'jquery',
        'lex!alarm',
        'dialogs',
        'nmodule/alarm/rc/Alarm',
        'bajaux/commands/Command',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/alarm/rc/fe/AlarmNotesEditor',
        'css!nmodule/alarm/rc/db/templates/view' ], function (
        baja,
        $,
        lexs,
        dialogs,
        Alarm,
        Command,
        feDialogs,
        AlarmNotesEditor) {

  'use strict';

  var LEX = lexs[0];

  /**
   * Adds notes to the selected alarm.
   *
   * @class
   * @alias module:nmodule/alarm/rc/db/commands/NotesCommand
   * @param {module:nmodule/alarm/rc/db/DatabaseMaintenance} view The alarm database
   * @param  {Array} [selectedRecords] An optional array of alarm records.
   * maintenance view.
   * @extends {module:bajaux/commands/Command}
   */
  var NotesCommand = function (view, selectedRecords) {
     Command.call(this, {
      module: 'alarm',
      lex: 'commands.notes',
      func: function () {
        var records;

        if (selectedRecords !== undefined) {
          records = selectedRecords;
        } else {
          records = view.$getCurrentRecords();
        }

        if (!records.length) {
          return dialogs.showOk(LEX.get('alarm.noRecordSelected')).promise();
        }

        var currentNotes = NotesCommand.$getCurrentNotes(records);

        return feDialogs.showFor({
          value: currentNotes,
          formFactor: 'mini',
          title: LEX.get('command.addNotes.label'),
          type: AlarmNotesEditor
        })
          .then(function (notes) {
            // Bail if the user hit cancel.
            if (notes === null) {
              return;
            }

            return baja.Ord.make('alarm:').get()
              .then(function (space) {
                // Add the notes to the server.
                return space.addNoteToAlarms({
                  ids: records.map(function (record) { return record.getUuid().encodeToString(); }),
                  notes: notes
                });
              })
              .then(function () {
                // Refresh the page after the notes have been added.
                if (view.pager) {
                  view.pager.update();
                }
              });
          });
      }
    });
  };

  /**
   * Return the current notes to use for the alarm records.
   *
   * @inner
   * @private
   *
   * @param  {Array} records An array of alarm records.
   * @returns {String} The current notes to display in any any UI.
   */
  NotesCommand.$getCurrentNotes = function (records) {
    var notes;
    switch(records.length) {
      case 0:
        notes = '';
        break;
      case 1:
        notes = records[0].getAlarmData().get('notes', '');
        break;
      default:
        notes = LEX.get('alarm.multipleAlarmNotes');
    }
    return notes;
  };

  NotesCommand.prototype = Object.create(Command.prototype);
  NotesCommand.prototype.constructor = NotesCommand;

  return NotesCommand;
});

