/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/SequenceCommand
 */
define([
  'baja!',
  'lex!alarm',
  'Promise',
  'jquery',
  'underscore' ], function (
  baja,
  lexs,
  Promise,
  $,
  _) {
  'use strict';

  var BATCH_SIZE = 10;

  function getSortColumn(view) {
    var table  = view.$table,
        header = table.jq().find('th').filter(function (index, elem) {
          var header = $(elem);
          return header.hasClass('asc') || header.hasClass('desc');
        }).first();

    return {
      column: header.length ? header.data('column').getName() : 'timestamp',
      desc: !header.length || !header.hasClass('asc')
    };
  }

  function fillBuffer(that) {
    return that.$support.getSingleSourceSummary({
      source: that.$source,
      timeRange: that.$timeRange,
      filterSet: that.$filterSet,
      offset: that.$offset,
      limit: BATCH_SIZE,
      column: that.$sortColumn.column,
      sortDesc: that.$sortColumn.desc
    }).then(function (summary) {
      that.$records = summary.records || [];
    });
  }

  function toSubjects(rows) {
    return _.invoke(rows, 'getSubject');
  }

  /**
   * Sequence Command
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view
   */
  var SequenceCommand = function (view) {
    this.$view = view;
    this.$viewModel = view.$getViewModel();
    this.$support = this.$viewModel.$getSupport();
  };

  SequenceCommand.prototype.constructor = SequenceCommand;

  /**
   *  Initializes the alarm record buffer and returns the current alarm subject
   *
   *  @return {Promise.<Object>} Promise resolved with current subject
   */
  SequenceCommand.prototype.init = function () {
    var ss         = this.$view.isSingleSourceView(),
        table      = this.$view.$getAlarmTable(),
        tableModel = table.getModel(),
        rows       = tableModel.getRows(),
        selected   = table.getSelectedRows(),
        subject    = selected[0].getSubject(),
        subjects   = toSubjects(rows);

    this.$source = subject.source;
    this.$sortColumn = getSortColumn(this.$view);
    this.$timeRange = this.$viewModel.$getTimeRange();
    this.$filterSet = this.$viewModel.$getFilterSet();
    this.$offset = ss ? this.$viewModel.$getPageOffset() : 0;
    this.$index = tableModel.getRowIndex(selected[0]);
    this.$records = subjects;
    return Promise.resolve(subject);
  };

  /**
   *  Returns the next logical alarm record in the sequence. This method will refill the alarm buffer
   *  if the index into the current buffer exceeds the buffer bounds
   *
   *  @return {Promise.<Object|null>} Promise resolved with next alarm subject or null if end of record set
   */
  SequenceCommand.prototype.next = function () {
    var that       = this,
        ss         = this.$view.isSingleSourceView(),
        table      = this.$view.$getAlarmTable(),
        tableModel = table.getModel(),
        rows       = tableModel.getRows(),
        records    = ss ? this.$records : toSubjects(rows),
        length     = (records && records.length) || 0,
        lastBuffer,
        lastOffset,
        lastLimit;

    if (length) {

      if (this.$index < length - 1) { //next record is in current buffer
        return Promise.resolve(records[++this.$index]);
      } else if (ss) { //refill buffer for single source view
        lastBuffer = records;
        lastOffset = this.$offset;
        lastLimit = length;
        this.$offset += lastLimit; //advance offset beyond current buffer

        return fillBuffer(this).then(function () {
          records = that.$records;
          if (records.length) {
            that.$index = 0;
            return records[0];
          } else { //restore last valid buffer
            that.$records = lastBuffer;
            that.$index = lastLimit - 1; //set index to last element of previous buffer
            that.$offset = lastOffset;
            return null;
          }
        });
      } else { //position index at last record in the main summary table
        this.$index = length - 1;
      }
    }

    return Promise.resolve();
  };

  /**
   *  Returns the prior logical alarm record in the sequence. This method will refill the alarm buffer
   *  if the index into the current buffer becomes negative
   *
   *  @return {Promise.<Object|null>} Promise resolved with previous alarm subject or null if end of record set
   */
  SequenceCommand.prototype.previous = function () {
    var that       = this,
        ss         = this.$view.isSingleSourceView(),
        table      = this.$view.$getAlarmTable(),
        tableModel = table.getModel(),
        rows       = tableModel.getRows(),
        records    = ss ? this.$records : toSubjects(rows),
        length     = (records && records.length) || 0,
        lastBuffer,
        offsetNegative;

    if (length) {
      if (this.$index > 0 && this.$index < length) { //previous record is in current buffer
        return Promise.resolve(records[--this.$index]);
      } else if (ss) { //refill buffer for single source view
        lastBuffer = this.$records;
        this.$offset -= BATCH_SIZE; //reset to previous offset
        offsetNegative = this.$offset < 0;
        this.$offset = offsetNegative ? 0 : this.$offset;
        this.$index = BATCH_SIZE - 1; //set index to last element of previous buffer
        return fillBuffer(this).then(function () {
          records = that.$records;
          if (records.length) {
            //a negative offset indicates the logical end of the record set, set the index to zero
            that.$index = offsetNegative ? 0 : records.length - 1;

            //if the first record of the new set equals the first record of last buffer
            //there's been no change so return null to disable the 'previous' command
            return lastBuffer.length && _.isEqual(lastBuffer[0], records[0]) ?
              null : records[that.$index];
          } else { //restore last valid buffer
            that.$index = 0; //set index to first element of last valid buffer
            that.$records = lastBuffer;
            return null;
          }
        });
      } else {
        this.$index = 0; //position index at first record in the main summary table
      }
    }
    return Promise.resolve();
  };

  /**
   *  Resets sequence state to defaults and clears record buffer
   *
   *  @return {Promise} Promise resolved when sequence state cleared
   */
  SequenceCommand.prototype.reset = function () {
    this.$offset = 0;
    this.$index = 0;
    this.$records = null;
    return Promise.resolve();
  };

  SequenceCommand.prototype.$getOffset = function () {
    return this.$offset;
  };

  SequenceCommand.prototype.$getIndex = function () {
    return this.$index;
  };

  SequenceCommand.prototype.$getRecords = function () {
    return this.$records;
  };

  return SequenceCommand;
});
