/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint browser:true*/

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/audio/AlarmAudioQueue
 */
define([], function () {
  'use strict';

  var DEFAULT_ALARM_DELAY = 10000;

  /**
   * Manages alarm sounds for the alarm console.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/audio/AlarmAudioQueue
   * @param {module:nmodule/alarm/rc/console/support/AlarmConsoleAudioSupport} audioSupport
   */
  var AlarmAudioQueue = function AlarmAudioQueue(audioSupport, wnd) {
    this.$timer = null;
    this.$queue = [];
    this.$audioSupport = audioSupport;
    this.$enabled = true;
    this.$delay = DEFAULT_ALARM_DELAY;
    this.$wnd = wnd || window;
  };

  /**
   * The default alarm delay (in milliseconds) to use.
   *
   * @type {Number} The delay time in milliseconds.
   */
  AlarmAudioQueue.DEFAULT_ALARM_DELAY = DEFAULT_ALARM_DELAY;

  /**
   * Set the time delay between playing alarms.
   *
   * @param {Number} The delay time.
   */
  AlarmAudioQueue.prototype.setDelay = function (delay) {
    var that = this;
    if (that.$delay !== delay) {
      that.$delay = delay;

      // Reschedule the play timer
      that.reschedulePlayTimer();
    }
  };

  /**
   * Return true if a sound is scheluled to be played in the not too distant future.
   *
   * @return {Boolean} Returns true if a sound is scheduled to be played.
   */
  AlarmAudioQueue.prototype.isPlayScheduled = function () {
    return this.$timer !== null;
  };

  /**
   * Play an alarm sound and if there's more in the queue, schedule the next
   * alarm sound to be played.
   */
  AlarmAudioQueue.prototype.play = function () {
    var that = this,
        sound;

    // Bail if we can't play any sounds.
    if (!that.isEnabled()) {
      return;
    }

    that.stopPlayTimer();

    sound = that.$queue.shift(); // Remove eldest element from queue.
    if (sound) {
      that.$audioSupport.play(sound);

      // If this is a continously sounding alarm then add the sound to the top of the queue
      // so it can be played again.
      if (that.isContinuous()) {
        that.$queue.push(sound);
      }
    }

    that.reschedulePlayTimer();
  };

  AlarmAudioQueue.prototype.reschedulePlayTimer = function () {
    var that = this;

    that.stopPlayTimer();

    // If there's anything left in the queue then set a timer to play the next sound.
    if (that.$queue.length) {
      that.$timer = that.$wnd.setTimeout(function () {
        that.play();
      }, that.$delay);
    }
  };

  /**
   * Stops any timers associated with playing sounds.
   */
  AlarmAudioQueue.prototype.stopPlayTimer = function () {
    var that = this;
    that.$wnd.clearTimeout(that.$timer);
    that.$timer = null;
  };

  /**
   * Stops all current sounds from playing.
   * This also clears any sounds about to be played.
   */
  AlarmAudioQueue.prototype.stopAll = function () {
    this.stopPlayTimer();
    this.$audioSupport.stopAll();
  };

  /**
   * Set whether the alarm queue is enabled or not.
   *
   * @param {Boolean} enabled True if enabled.
   */
  AlarmAudioQueue.prototype.setEnabled = function (enabled) {
    this.$enabled = enabled;

    if (!enabled) {
      this.stopAll();
    }
  };

  /**
   * Return true if the queue is enabled.
   */
  AlarmAudioQueue.prototype.isEnabled = function () {
    return this.$enabled;
  };

  /**
   * Set whether the alarm queue plays a sound continuously or not.
   *
   * @param {Boolean} continuous True if continuous.
   */
  AlarmAudioQueue.prototype.setContinuous = function (continuous) {
    this.$continuous = continuous;
  };

  /**
   * Return true if the queue plays a sound continuously or not.
   */
  AlarmAudioQueue.prototype.isContinuous = function () {
    return this.$continuous;
  };

  /**
   * Add the sound to the queue to be played. Please note, this will not
   * play a sound. A sound will only be added if the queue is enabled.
   *
   * @param {String} sound The sound to be played.
   */
  AlarmAudioQueue.prototype.add = function (sound) {
    if (this.isEnabled()) {
      this.$queue.push(sound);
    }
  };

  /**
   * Search the queue and remove only the first matching sound found.
   *
   * @param  {String} sound The sound to be removed.
   */
  AlarmAudioQueue.prototype.removeFirst = function (sound) {
    var queue    = this.$queue,
        toRemove = -1,
        i;

    // Remove the first matching sound we find.
    for (i = 0; i < queue.length; ++i) {
      if (queue[i] === sound) {
        toRemove = i;
        break;
      }
    }

    if (toRemove > -1) {
      queue.splice(toRemove, 1);
    }
  };

  return AlarmAudioQueue;
});
