/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/baja/AlarmPriorities
 */
define(['baja!',
  'lex!alarm',
  'bajaScript/baja/obj/objUtil',
  'underscore'], function (baja,
                           lexs,
                           objUtil,
                           _) {

  'use strict';

  var Simple       = baja.Simple,
      cacheDecode  = objUtil.cacheDecode,
      cacheEncode  = objUtil.cacheEncode,

      alarmLex     = lexs[0],

      MIN_PRIORITY = 0,
      MAX_PRIORITY = 255,

      PROP_NAMES   = ['toOffnormal', 'toFault', 'toNormal', 'toAlert'],
      PAIR_REGEX   = /(\w+)=(\d+)/;

  function checkPriority(priority) {
    if (priority < MIN_PRIORITY || priority > MAX_PRIORITY) {
      throw new Error(alarmLex.get({
        key: 'alarmPriorities.outOfRange',
        args: [priority, MIN_PRIORITY, MAX_PRIORITY]
      }));
    }
    return priority || 0;
  }

  /**
   * BajaScript representation of a `alarm:AlarmPriorities` value.
   *
   * @class
   * @extends baja.Simple
   * @alias module:nmodule/alarm/rc/baja/AlarmPriorities
   */
  var AlarmPriorities = function AlarmPriorities() {
    Simple.apply(this, arguments);
  };
  AlarmPriorities.prototype = Object.create(Simple.prototype);
  AlarmPriorities.prototype.constructor = AlarmPriorities;

  AlarmPriorities.MAX_PRIORITY = MAX_PRIORITY;
  AlarmPriorities.MIN_PRIORITY = MIN_PRIORITY;

  /**
   * Create a new `alarm:AlarmPriorities` instance.
   *
   * @param {number} toOffNormal
   * @param {number} toFault
   * @param {number} toNormal
   * @param {number} toAlert
   * @returns {module:nmodule/alarm/rc/baja/AlarmPriorities}
   */
  AlarmPriorities.make = function (toOffNormal, toFault, toNormal, toAlert) {
    var ap = new AlarmPriorities();
    ap.$toOffnormal = checkPriority(toOffNormal);
    ap.$toFault = checkPriority(toFault);
    ap.$toNormal = checkPriority(toNormal);
    ap.$toAlert = checkPriority(toAlert);
    return ap;
  };

  /**
   * @see .make
   * @returns {module:nmodule/alarm/rc/baja/AlarmPriorities}
   */
  AlarmPriorities.prototype.make = function () {
    return AlarmPriorities.make.apply(AlarmPriorities, arguments);
  };

  /**
   * Decode an `AlarmPriorities` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/alarm/rc/baja/AlarmPriorities}
   */
  AlarmPriorities.prototype.decodeFromString = cacheDecode(function (str) {
    var args = [0, 0, 0, 0];
    _.each(str.split(';'), function (pair) {
      var result = PAIR_REGEX.exec(pair);
      if (result) {
        var i = _.indexOf(PROP_NAMES, result[1]);
        if (i >= 0) {
          args[i] = parseInt(result[2], 10);
        }
      }
    });
    return AlarmPriorities.make.apply(AlarmPriorities, args);
  });

  /**
   * Encode this `AlarmPriorities` to a string.
   *
   * @function
   * @returns {String}
   */
  AlarmPriorities.prototype.encodeToString = cacheEncode(function () {
    var that = this;
    return _.map(PROP_NAMES, function (arg) {
      return arg + '=' + that['$' + arg];
    }).join(';');
  });

  /**
   * Get the priority assigned to the toOffnormal transition type.
   * @return {number} The toOffnormal priority.
   */
  AlarmPriorities.prototype.getToOffnormal = function () {
    return this.$toOffnormal;
  };

  /**
   * Get the priority assigned to the toFault transition type.
   * @return {number} The toFault priority.
   */
  AlarmPriorities.prototype.getToFault = function () {
    return this.$toFault;
  };

  /**
   * Get the priority assigned to the toNormal transition type.
   * @return {number} The toNormal priority.
   */
  AlarmPriorities.prototype.getToNormal = function () {
    return this.$toNormal;
  };

  /**
   * Get the priority assigned to the toAlert transition type.
   * @return {number} The toAlert priority.
   */
  AlarmPriorities.prototype.getToAlert = function () {
    return this.$toAlert;
  };

  AlarmPriorities.DEFAULT = AlarmPriorities.make(255, 255, 255, 255);

  baja.registerType('alarm:AlarmPriorities', function () {
    return AlarmPriorities;
  });

  return AlarmPriorities;
});
