/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

define([
  'baja!',
  'baja!alarm:AlarmRecord,box:AlarmChannel,alarm:AlarmConsoleOptions',
  'Promise',
  'jquery',
  'underscore'
], function (baja,
             types,
             Promise,
             $,
             _) {
  'use strict';


  /**
   * API Status: **Private**
   * @exports module:nmodule/alarm/rc/Alarm
   */
  var Alarm = {};

  /**
   * Asynchronously retrieve alarms for a given dynamic time range
   *
   * @param {object} params
   * @param {baja.Simple} params.timeRange - `bql.DynamicTimeRange` within which to query for results
   * @param {number} [params.limit=-1] - Maximum number of alarms to retrieve
   * @param {number} [params.offset=-1] - Starting offset of result cursor
   * @param {baja.Component} [params.filterSet] - `bql.FilterSet` to apply to query
   * @param {Boolean} [params.sortDescending=false] - Retrieve results in descending order
   * @param {String} [params.columnName=timestamp] - columnName on which to order results
   * @param {Object} params.alarmClassDisplayNames - Map of alarm class names to display names
   * @param {baja.TimeZoneDatabase} params.timeZoneDatabase - Local timezone database for resolving Time Zone ids.
   * @param {number} [params.chunkSize=params.limit] only query this number of alarm
   * records at a time, to avoid overloading the station with a request too large.
   * @returns {Promise.<{ alarms: baja.Complex[], count: number }>} resolved with array of
   * `alarm:AlarmRecord`s and an alarm count
   */
  Alarm.retrieveAlarms = function (params) {
    // keep support for old argument list
    if (baja.hasType(params, 'bql:DynamicTimeRange')) {
      var args = arguments;
      params = {
        timeRange: args[0],
        limit: args[1],
        offset: args[2],
        filterSet: args[3],
        sortDescending: args[4],
        columnName: args[5],
        alarmClassDisplayNames: args[6],
        timeZoneDatabase: args[7],
        chunkSize: args[1]
      };
    }

    return getAlarmSpace()
      .then(function (space) {
        return pageAlarmRecords(space, params);
      })
      .then(function (result) {
        return decodeAlarms(params.alarmClassDisplayNames, params.timeZoneDatabase, result.r)
          .then(function (alarms) {
            return {
              alarms: alarms,
              count: result.c
            };
          });
      });
  };

  /**
   * Pull down the records from the station, using a chunk size to avoid
   * requesting a kabillion records at once.
   * @param {baja.AlarmSpace} space
   * @param {object} params
   * @returns {Promise.<{ r: string[], c: number }>} the retrieved BSON-encoded
   * records, out of how many total records were available
   */
  function pageAlarmRecords(space, params) {
    var timeRange = params.timeRange,
      limit = params.limit,
      offset = params.offset,
      filterSet = params.filterSet,
      sortDescending = params.sortDescending,
      columnName = params.columnName,
      chunkSize = params.chunkSize || limit,
      count,
      records = [];

    return doPage(offset);

    function doPage(startRecord) {
      return space.queryAlarmDatabase(timeRange, chunkSize, startRecord, filterSet, sortDescending, columnName)
        .then(function (result) {
          var returnedRecords = result.r;

          count = result.c;
          records = records.concat(returnedRecords.slice(0, chunkSize));

          if (returnedRecords.length >= chunkSize && records.length < limit) {
            return doPage(startRecord + chunkSize);
          } else {
            return { r: records.slice(0, limit), c: count };
          }
        });
    }
  }

  function decodeAlarms(alarmClassMap, tzDatabase, records) {
    return Promise.all(_.map(records, function (record) {
      return baja.bson.decodeAsync(record, baja.$serverDecodeContext)
        .then(function (decoded) {
          return Alarm.$updateRecord(alarmClassMap, tzDatabase, decoded);
        });
    }));
  }

  /**
   * Update the AlarmRecord with the correct baja.TimeZone and Alarm Class Display Name
   * as well as adding properties for source and ack state values.
   * @private
   * @param {object} alarmClassDisplayNames
   * @param {baja.TimeZoneDatabase} tzDatabase
   * @param {baja.Struct} rec `alarm:AlarmRecord`
   */
  Alarm.$updateRecord = function (alarmClassDisplayNames, tzDatabase, rec) {
    rec.alarmData = _.mapObject(rec.getAlarmData().toObject(), function (val, key) {
        switch (key) {
          case 'TimeZone':
            return tzDatabase.getTimeZone(val.getId());
          default:
            return val instanceof baja.Simple ? val.encodeToString() : val;
        }
      });

    var alarmClass = rec.getAlarmClass(); // 'alarmClass' is the display name for the AlarmClass, used for display in the console
    rec.setAlarmClass(alarmClassDisplayNames[alarmClass] || alarmClass);
    rec._alarmClass = alarmClass; // '_alarmClass' is the name for the AlarmClass, used as key for permissions map
    rec._sourceState = rec.getSourceState().getTag();
    rec._ackState = rec.getAckState().getTag();
    return rec;
  };

/////////////////////////////////////////////////////////////////////
//Alarm Database Maintenance
/////////////////////////////////////////////////////////////////////

  /**
   * Retrieve the number of alarms in the database for a given time range (null arg returns total alarms)
   * @returns {Promise} Promise resolved with the alarm count
   */
  Alarm.getAlarmCount = function (timeRange, filterSet) {
    return getAlarmSpace().then(function (space) {
      return space.getRecordCount(timeRange, filterSet);
    });
  };

  /**
   * Asynchronously clear all alarms in the alarm database
   * @returns {Promise} Promise
   */
  Alarm.clearAllRecords = function (callbacks) {
    return getAlarmSpace().then(function (space) {
      return space.clearAllRecords({
        ok: function (response) {
          callbacks.ok(response);
        },

        fail: function (err) {
          callbacks.fail(err);
        }
      });
    });
  };

  /**
   * Asynchronously clear alarms records that are older than the given
   * timestamp
   *
   * @param {baja.AbsTime} [beforeTime] Records with timestamps older than this timestamp will be cleared
   * @returns {Promise} Promise
   * @param callbacks
   */
  Alarm.clearOldRecords = function (beforeTime, callbacks) {
    return getAlarmSpace().then(function (space) {
      return space.clearOldRecords(beforeTime, {
        ok: function (response) {
          callbacks.ok(response);
        },

        fail: function (err) {
          callbacks.fail(err);
        }
      });
    });
  };

  /**
   * Asynchronously clear the alarm records specified by the uuids
   *
   * @param {Array} uuids  Array of encoded baja.Uuid strings
   * @returns {Promise} Promise
   * @param callbacks
   */
  Alarm.clearRecords = function (uuids, callbacks) {
    return getAlarmSpace().then(function (space) {
      return space.clearRecords(uuids, {
        ok: function (response) {
          callbacks.ok(response);
        },

        fail: function (err) {
          callbacks.fail(err);
        }
      });
    });
  };

  /**
   * Asynchronously add a note to an alarm record
   * @param alarm The alarm record to which to add the new note
   * @param note
   * @param callbacks
   * @returns {Promise} Promise
   */
  Alarm.addNote = function (alarm, note, callbacks) {
    return getAlarmSpace().then(function (space) {
      return space.addNote(alarm, note, {
        ok: function (response) {
          callbacks.ok(response);
        },

        fail: function (err) {
          callbacks.fail(err);
        }
      });
    });
  };

  /**
   * Asynchronously retrieves a map of alarm class names to alarm class display names
   * @returns {Promise} Promise resolved with object map
   */
  Alarm.getDisplayNamesMap = function () {
    return getAlarmSpace().then(function (space) {
      return space.getDisplayNamesMap();
    });
  };

  /**
   * Asynchronously retrieves a map of alarm class names to alarm class permissions
   *
   * @since Niagara 4.9
   *
   * @params {Array.<String>} alarmClasses alarm class names for displayed alarms
   * @returns {Promise} Promise resolved with object map
   */
  Alarm.getPermissionsMap = function (alarmClasses) {
    return getAlarmSpace().then(function (space) {
      return space.getPermissionsMap(alarmClasses);
    });
  };

  function getAlarmSpace() { return baja.Ord.make('alarm:').get(); }

  return Alarm;
});
