/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/menu/ReportListView
 */
define(['baja!', 'baja!baja:Link', 'lex!bajaux,webEditors', 'jquery', 'Promise', 'underscore', 'moment', 'dialogs', 'nmodule/analytics/ext/datatables/jquery.dataTables', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'bajaux/Widget', 'nmodule/analytics/rc/util/SimplePropertySheet', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/report/util/reportConstants', 'bajaux/dragdrop/dragDropUtils', 'nmodule/analytics/rc/report/util/reportUtils', 'hbs!nmodule/analytics/rc/report/menu/reportListTemplate', 'hbs!nmodule/analytics/rc/report/menu/reportTableTemplate', 'lex!analytics', 'css!nmodule/analytics/rc/report/styles/reportStyles', 'css!nmodule/analytics/ext/datatables/jquery.dataTables'], function (baja, types, lexs, $, Promise, _, moment, dialogs, dataTable, fe, feDialogs, Widget, SimplePropertySheet, analyticsUtil, reportConstants, dragDropUtils, reportUtils, reportListTemplate, reportTableTemplate, lexicon) {

    'use strict';

    var lex = lexicon[0],
        hlClass = 'highligted-table-cont';

    /**
     * ReportListView
     * @class
     * @alias module:nmodule/analytics/rc/report/menu/ReportListView
     * @extends module:bajaux/Widget
     * @param params
     */
    var ReportListView = function ReportListView(params) {
        params = params || {};
        var that = this;
        Widget.call(that, params.moduleName || 'analytics', params.keyName, params.formFactor);
    };

    ReportListView.prototype = Object.create(Widget.prototype);
    ReportListView.prototype.constructor = ReportListView;

    /**
     * Initialize the reports
     */
    ReportListView.prototype.doInitialize = function (dom) {
        this.columnWidths = [];
        this.folderList = [];
        this.orderBy = {
            repName: "asc",
            lastUpdated: "desc"
        };
        this.isWb = window.niagara && window.niagara.env && window.niagara.env.type === "wb";
        var that = this;
        dom.html(reportListTemplate({
            allReports: lex.get("report.list.view"),
            newReport: lex.get("report.new.title"),
            luLabel: lex.get("report.label.lastupdate"),
            rnLabel: lex.get("report.label.repname"),
            reportOrder: lex.get("report.label.order"),
            reportSearch: lex.get("report.label.search")
        }));

        var width = 100 / reportConstants.DEFAULT_REP_LIST_COL_COUNT;
        for (var i = 0; i < reportConstants.DEFAULT_REP_LIST_COL_COUNT; i++) {
            this.columnWidths.push({ "width": width + "%" });
        }
        // Fetch all the folders accessible by the users.
        that.$registerNewReport();
        that.$registerOrderChange();
        that.$registerDragDropHandler();
    };

    /**
     * Loads the component. If the component is mounted and has actions to fire,
     * the "actions" command will be enabled.
     *
     * @param {baja.Complex} complex
     * @returns {Promise} promise to be resolved when the complex has
     * loaded and the actions command has been enabled or disabled.
     */
    ReportListView.prototype.doLoad = function (reportFolder) {
        var that = this;
        var repList = [],
            columnWidths = that.columnWidths,
            counter = 0,
            repTotal = 0,
            sortBy = "repName",
            rp = reportFolder.getSlotPath().toString();
        sortBy = $("select.order-report", this.jq()).val();
        var folderListPromise = this.getApplicableFolders(rp);
        folderListPromise.then(function () {
            return baja.Ord.make("station:|" + rp + "|bql:select * from analytics:AnalyticReport order by " + sortBy + " " + that.orderBy[sortBy]).get({
                cursor: {
                    each: function each(report, index) {
                        repTotal += 1;
                        if (index % reportConstants.DEFAULT_REP_LIST_COL_COUNT === 0) {
                            repList[counter++] = [];
                        }
                        // We get table element as matrix. Iterate and populate the report list array
                        var hyperlinkLocation = "station:|" + report.get("slotPath") + "|view:analytics:AnalyticUxReportView";
                        if (window.niagara && window.niagara.env.type !== "wb") {
                            hyperlinkLocation = "/ord/" + hyperlinkLocation;
                        }
                        var n = baja.SlotPath.unescape(report.get("repName"));
                        var lu = that.getReportUpdateTime(report.get("lastUpdated"));
                        repList[counter - 1].push({
                            repName: n,
                            repLink: hyperlinkLocation,
                            repDesc: report.get("description") || n,
                            repLoc: report.get("location").getTag(),
                            repLocDn: lex.get({ key: "report.location.label",
                                args: [baja.SlotPath.unescape(report.get("location").getTag())] }),
                            repClass: that.getReportIcon(report.get("reportType").getOrdinal()),
                            lastUpdate: lex.get({ key: "report.lastupdated.time", args: [lu] })
                        });
                    },
                    after: function after() {
                        that.$padRepList(repList);
                    },
                    limit: -1
                }
            }).then(function () {
                return Promise.resolve(repList);
            }).then(function () {
                var search = $("input.rep-srch", that.jq()).val();
                if (search && search.length > 0) {
                    that.renderFilteredResults(repList, search, 300);
                } else {
                    that.renderReportList(repList, columnWidths, repTotal);
                    that.$registerReportSearch(repList);
                }
                return Promise.resolve(reportFolder);
            });
        });
    };

    /**
     * Destroy the table before delegating to parent
     */
    ReportListView.prototype.doDestroy = function () {
        var that = this,
            args = arguments;
        this.destroyReportList().then(function () {
            return Widget.prototype.doDestroy.apply(that, args);
        });
    };

    /**
     * Destroy the report list
     */
    ReportListView.prototype.destroyReportList = function () {
        var jq = this.jq();
        var table = $('.reports-table', jq),
            container = $(".report-table-container", jq);
        if ($.fn.DataTable.isDataTable(table)) {
            table.DataTable().destroy();
        }
        container.children().off();
        container.empty();
        return Promise.resolve(true);
    };

    /**
     * Render the list of reports.
     * @param repList
     * @param columnWidths
     * @param repTotal
     */
    ReportListView.prototype.renderReportList = function (repList, columnWidths, repTotal) {
        var that = this,
            jq = that.jq();
        var tableContainer = $(".report-table-container", jq);
        tableContainer.html(reportTableTemplate({
            reports: repList,
            isWb: that.isWb
        }));
        $("table.reports-table", that.jq()).DataTable({
            "columns": columnWidths,
            'dom': 'lfipt',
            "language": {
                "emptyTable": lex.getSafe("report.list.empty"),
                "oPaginate": {
                    "sPrevious": '<i class="icon-icons-x16-arrowLeft" ></i>',
                    "sNext": '<i class="icon-icons-x16-arrowRight" ></i>'
                }
            },
            "drawCallback": function drawCallback(settings) {
                that.$registerReportEvents();
            },
            "infoCallback": function infoCallback(settings, start, end, max, total, pre) {
                if (total > 0) {
                    var colCount = reportConstants.DEFAULT_REP_LIST_COL_COUNT;
                    var repEnd = end * colCount;
                    if (end === total) {
                        if (repTotal % colCount === 0) {
                            repEnd = end * reportConstants.DEFAULT_REP_LIST_COL_COUNT;
                        } else if (end > 1) {
                            repEnd = (end - 1) * reportConstants.DEFAULT_REP_LIST_COL_COUNT + repTotal % colCount;
                        } else {
                            repEnd = repTotal % colCount;
                        }
                    }
                    return lex.getSafe({
                        "key": "report.list.message", args: [(start - 1) * reportConstants.DEFAULT_REP_LIST_COL_COUNT + 1, repEnd, repTotal]
                    });
                }
            },
            "pagingType": "simple",
            "pageLength": reportConstants.DEFAULT_REP_LIST_ROW_COUNT,
            "lengthChange": false,
            "searching": false,
            "ordering": false
        });
    };

    /**
     * Get the last report update time
     * @param updateTime
     */
    ReportListView.prototype.getReportUpdateTime = function (updateTime) {
        return moment(updateTime.getJsDate()).format(reportConstants.LASTUPDATE_TIMESTAMP_FORMAT);
    };
    /**
     * Get the report type
     * @param repType
     */
    ReportListView.prototype.getReportIcon = function (repType) {
        if (repType === 0) {
            return "rep-spectrum";
        } else if (repType === 1) {
            return "rep-loadduration";
        } else if (repType === 2) {
            return "rep-avgprof";
        } else if (repType === 3) {
            return "rep-aggregation";
        } else if (repType === 4) {
            return "rep-equipop";
        } else if (repType === 5) {
            return "rep-ranking";
        } else if (repType === 6) {
            return "rep-relcontrib";
        } else {
            return "";
        }
    };

    /**
     * Get the list of folders the user is privy to
     * @param rp
     * @returns {Promise}
     */
    ReportListView.prototype.getApplicableFolders = function (rp) {
        var that = this;
        var folderListPromise = baja.Ord.make("station:|" + rp + "|bql:select name,displayName from baja:Folder").get({
            cursor: {
                before: function before() {
                    that.folderList = [];
                },
                each: function each(folder, index) {
                    that.folderList.push({
                        n: folder.get("name"),
                        dn: folder.get("displayName")
                    });
                }
            }, lease: true
        });
        return folderListPromise;
    };
    /**
     * Register event for creation of new reports.
     */
    ReportListView.prototype.$registerReportEvents = function () {
        $("div.report-table-container").children().off();
        this.$registerDeleteReport();
        this.$registerEditReport();
        this.$registerCloneReport();
        this.$registerNavigationEvent();
    };

    /**
     * Register navigation event for workbench
     */
    ReportListView.prototype.$registerNavigationEvent = function () {
        var jq = this.jq();
        $("div.report-link-item", jq).click(function () {
            var hyperlinkLocation = $(this).attr("link");
            window.niagara.env.hyperlink(hyperlinkLocation);
        });
    };
    /**
     * Register event for creation of new reports.
     */
    ReportListView.prototype.$registerEditReport = function () {
        var jq = this.jq(),
            that = this;
        $("div.report-edit", jq).click(function (e) {
            var repFolder = that.value();
            var repHandle = $(this).parent(),
                repName = baja.SlotPath.escape(repHandle.attr("rn")),
                location = baja.SlotPath.escape(repHandle.attr("rl"));
            return repFolder.loadSlots().then(function () {
                var repLocation = repFolder.get(location);
                return repLocation.loadSlots().then(function () {
                    var rep = repLocation.get(repName);
                    return rep.lease().then(function () {
                        var param = baja.$("analytics:AnalyticReportParam", {
                            repName: rep.getRepName(),
                            description: rep.getDescription(),
                            reportType: rep.getReportType()
                        });
                        param.setFlags({
                            slot: "location",
                            flags: baja.Flags.HIDDEN
                        });
                        param.setFlags({
                            slot: "reportType",
                            flags: baja.Flags.HIDDEN
                        });
                        return feDialogs.showFor({
                            title: lex.get("report.edit"),
                            value: param,
                            type: SimplePropertySheet
                        }).then(function (trigger) {
                            if (trigger) {
                                rep.invoke({
                                    slot: "updateReport",
                                    value: param
                                }).then(function () {
                                    return that.destroyReportList().then(function (ret) {
                                        return that.load(repFolder);
                                    });
                                }).catch(function (err) {
                                    dialogs.showOk(err);
                                });
                            }
                        });
                    });
                });
            });
        });
    };

    /**
     * Register event for duplication of existing report.
     */
    ReportListView.prototype.$registerCloneReport = function () {
        var jq = this.jq(),
            that = this;
        $("div.report-clone", jq).click(function (e) {
            var repFolder = that.value();
            var repHandle = $(this).parent(),
                repName = baja.SlotPath.escape(repHandle.attr("rn")),
                location = baja.SlotPath.escape(repHandle.attr("rl"));
            return repFolder.loadSlots().then(function () {
                var repLocation = repFolder.get(location);
                return repLocation.loadSlots().then(function () {
                    var rep = repLocation.get(repName);
                    return rep.lease().then(function () {
                        var param = baja.$("analytics:AnalyticReportParam", {
                            repName: lex.get({ key: "report.name.clone", args: [rep.getRepName()] }),
                            description: rep.getDescription(),
                            reportType: rep.getReportType(),
                            location: rep.getLocation(),
                            filePath: rep.getFilePath()
                        });
                        param.setFlags({
                            slot: "reportType",
                            flags: baja.Flags.HIDDEN
                        });
                        param.setFlags({
                            slot: "filePath",
                            flags: baja.Flags.HIDDEN
                        });
                        return feDialogs.showFor({
                            title: lex.get({ key: "report.clone", args: [rep.getRepName()] }),
                            value: param,
                            type: SimplePropertySheet
                        }).then(function (trigger) {
                            if (trigger) {
                                repFolder.invoke({
                                    slot: "createNewReport",
                                    value: param
                                }).then(function () {
                                    return that.destroyReportList().then(function (ret) {
                                        return that.load(repFolder);
                                    });
                                }).catch(function (err) {
                                    dialogs.showOk(err);
                                });
                            }
                        });
                    });
                });
            });
        });
    };
    /**
     * Register event for ordering of exiting reports.
     */
    ReportListView.prototype.$registerOrderChange = function () {
        var jq = this.jq(),
            that = this;
        $("select.order-report", jq).change(function (e) {
            return that.destroyReportList().then(function (ret) {
                return that.load(that.value());
            });
        });
    };
    /**
     * Register event for creation of new report
     */
    ReportListView.prototype.$registerReportSearch = function (repList) {
        var jq = this.jq(),
            that = this;
        var elem = $("input.rep-srch", jq);
        elem.off();
        elem.keyup(function (e) {
            var search = elem.val();
            // if(search.length < 3) return;
            that.renderFilteredResults(repList, search, 200);
        });
    };
    /**
     * Rendere filtered results
     * @param repList
     * @param search
     * @param delay
     */
    ReportListView.prototype.renderFilteredResults = function (repList, search, delay) {
        search = search.toLowerCase();
        var finalArray = [],
            that = this;
        _.debounce(function () {
            var filteredList = [];
            _.each(repList, function (repRow) {
                var f = _.filter(repRow, function (rep) {
                    return rep.repName && rep.repName.toLowerCase().indexOf(search) > -1 || rep.repDesc && rep.repDesc.toLowerCase().indexOf(search) > -1;
                });
                Array.prototype.push.apply(filteredList, f);
            });
            var counter = 0,
                len = filteredList.length,
                index = 0;
            while (counter < len) {
                if (counter % reportConstants.DEFAULT_REP_LIST_COL_COUNT === 0) {
                    finalArray[index++] = [];
                }
                finalArray[index - 1].push(filteredList[counter]);
                counter++;
            }
            that.$padRepList(finalArray);
            return that.destroyReportList().then(function () {
                that.renderReportList(finalArray, that.columnWidths, filteredList.length);
                that.$registerReportSearch(repList);
                that.$registerReportEvents();
            });
        }, delay || 750)();
    };
    /**
     * Register event for creation of new report
     */
    ReportListView.prototype.$registerNewReport = function () {
        var jq = this.jq(),
            that = this;
        $("button.create-report", jq).click(function (e) {
            e.preventDefault();
            return that.$handleNewReportCreation();
        });
    };
    /**
     * Handle new report creation
     * @returns {*}
     */
    ReportListView.prototype.$handleNewReportCreation = function () {
        var that = this,
            repF = that.value();
        return repF.getActionParameterDefault({
            slot: "createNewReport"
        }).then(function (param) {
            param.setFlags({
                slot: "filePath",
                flags: baja.Flags.HIDDEN
            });
            param.setLocation(reportUtils.getFolderListEnum(that.folderList, 0));
            return feDialogs.showFor({
                title: lex.get("report.new"),
                value: param,
                type: SimplePropertySheet
            }).then(function (trigger) {
                if (trigger) {
                    return repF.invoke({
                        slot: "createNewReport",
                        value: param
                    }).then(function () {
                        return that.destroyReportList().then(function (ret) {
                            return that.load(repF);
                        });
                    }).catch(function (err) {
                        dialogs.showOk(err);
                    });
                }
            });
        });
    };
    /**
     * Register event for deletion of existing report
     */
    ReportListView.prototype.$registerDeleteReport = function () {
        var jq = this.jq(),
            that = this;
        $("div.report-delete", jq).click(function (e) {
            e.preventDefault();
            var repHandle = $(this).parent(),
                repName = baja.SlotPath.escape(repHandle.attr("rn")),
                location = baja.SlotPath.escape(repHandle.attr("rl"));
            return dialogs.showOkCancel(lex.getSafe("report.delete.confirm")).ok(function () {
                var repFolder = that.value();
                return repFolder.loadSlots().then(function () {
                    var repLocation = repFolder.get(location);
                    return repLocation.loadSlots().then(function () {
                        var rep = repLocation.get(repName);
                        return rep.lease().then(function () {
                            var param = baja.$("analytics:AnalyticReportParam", {
                                repName: repName,
                                location: rep.getLocation()
                            });
                            return repFolder.invoke({
                                slot: "deleteReport",
                                value: param
                            }).then(function () {
                                return that.destroyReportList().then(function (ret) {
                                    return that.load(repFolder);
                                });
                            });
                        });
                    });
                });
            });
        });
    };

    /**
     * Handle drag and drop of report from palette.
     */
    ReportListView.prototype.$registerDragDropHandler = function () {
        var that = this;
        var container = $('.report-table-container', this.jq());
        container.on('dragover dragenter', function (e) {
            e.preventDefault();
            container.addClass(hlClass);
        }).on('drop', function (e) {
            container.removeClass(hlClass);
            var clipboard = e.originalEvent.dataTransfer;
            dragDropUtils.fromClipboard(clipboard).then(function (envelope) {
                return envelope.toValues();
            }).then(function (values) {
                if (values.length !== 1) {
                    return false;
                }
                var repShadow = values[0];
                var repFolder = that.value();
                repShadow.getNavOrd().get({}).then(function (repPaletteItem) {
                    repFolder.getActionParameterDefault({ slot: "createNewReport" }).then(function (param) {
                        param.setReportType(reportUtils.getReportTypeFromJS(repPaletteItem.getJs().encodeToString()));
                        param.setFlags({
                            slot: "reportType",
                            flags: baja.Flags.READONLY
                        });
                        param.setFlags({
                            slot: "filePath",
                            flags: baja.Flags.HIDDEN
                        });
                        return feDialogs.showFor({
                            title: lex.get("report.palette.create"),
                            value: param,
                            type: SimplePropertySheet
                        }).then(function (trigger) {
                            if (trigger) {
                                repFolder.invoke({
                                    slot: "createNewReport",
                                    value: param
                                }).then(function () {
                                    return that.destroyReportList().then(function (ret) {
                                        return that.load(repFolder);
                                    });
                                });
                            }
                        });
                    });
                });
            });
            return false;
        }).on("dragleave", function (e) {
            container.removeClass(hlClass);
        });
    };

    /**
     * Pad the report list with dummy columns
     */
    ReportListView.prototype.$padRepList = function (repList) {
        _.each(repList, function (repRow) {
            for (var i = repRow.length; i < reportConstants.DEFAULT_REP_LIST_COL_COUNT; i++) {
                repRow[i] = { repName: false };
            }
        });
    };

    return ReportListView;
});
