/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/OrdChooserWithHierarchy
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/links/OrdChooser', 'nmodule/webEditors/rc/wb/tree/stationTree', 'nmodule/webEditors/rc/wb/tree/NavTree', 'nmodule/webEditors/rc/wb/tree/BajaNavTreeNode', 'nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'lex!analytics'], function (baja, Promise, _, fe, OrdChooser, stationTree, NavTree, BajaNavTreeNode, BajaComplexTreeNode, spaceUtils, lexs) {

  'use strict';

  var lex = lexs[0];

  /**
   * Given a path of node names (gotten from the component tree when a node
   * is selected), convert it to the corresponding hierarchy ORD.
   *
   * @inner
   * @param {Array.<String>} path
   * @returns {baja.Ord|undefined} ORD derived from the selected node's path,
   * or undefined if no actual ORD is selected
   */
  function hierarchyOrdToOrd(path) {
    var ordStr = path[1]; //start from index 1 - skip root node from stationTree
    switch (ordStr) {
      case 'hierarchy':
        ordStr = 'hierarchy:';
        break;
    }
    if (!ordStr) {
      return baja.Ord.DEFAULT;
    } else {
      for (var oIndex = 2; oIndex < path.length; oIndex++) {
        ordStr += "/" + baja.SlotPath.escape(path[oIndex]);
      }
      return baja.Ord.make(ordStr).relativizeToSession();
    }
  }

  /**
   * Given an ORD, convert it to a path of node names. This will be used to
   * preselect a node in the component tree.
   *
   * @inner
   * @param {baja.Ord} ord
   * @param {baja.Component} ordBase
   * @returns {Promise.<Array.<String>>}
   */
  function ordToPath(ord) {
    var body = stationTree.SESSION_NODE_NAME;

    if (ord.equals(baja.Ord.DEFAULT)) {
      return Promise.resolve([body]);
    }

    var query = ord.relativizeToSession().parse(),
        slotPath = query.get('slot'),
        file = query.get('file'),
        hierarchy = query.get('hierarchy');

    if (slotPath) {
      return baja.Ord.make('station:|slot:/').get().then(function (station) {
        return [body, String(station.getNavOrd().relativizeToSession())].concat(slotPath.getBody().substring(1).split('/'));
      });
    } else if (file) {
      //strip leading "^" - it's assumed we're looking in the file root
      body += '/file/' + file.getBody().replace(/^\^/, '');
    } else if (hierarchy) {
      var hPath = hierarchy.getBody().substring(1).split('/');
      var ret = [body, "hierarchy"];
      for (var hIndex = 0; hIndex < hPath.length; hIndex++) {
        ret.push(baja.SlotPath.unescape(hPath[hIndex]));
      }
      return Promise.resolve(ret);
    }
    return Promise.resolve(body.split('/'));
  }

  /**
   * AnalyticOrdChooser field editor is used to select an ORD for use in/as analytic node
   * @class
   * @alias module:nmodule/analytics/rc/report/fe/OrdChooserWithHierarchy
   * @extends module:nmodule/webEditors/rc/wb/links/OrdChooser
   * @param params
   */
  var OrdChooserWithHierarchy = function OrdChooserWithHierarchy(params) {
    OrdChooser.apply(this, arguments);
  };

  OrdChooserWithHierarchy.prototype = Object.create(OrdChooser.prototype);

  OrdChooserWithHierarchy.prototype.constructor = OrdChooserWithHierarchy;

  /**
   * Override the default ORD chooser to show nodes from hierarchy as well
   * @param {jQuery} dom
   */
  OrdChooserWithHierarchy.prototype.$loadNavTree = function (ord) {
    var that = this;
    return Promise.all([OrdChooser.prototype.$loadNavTree.apply(this, arguments)]).then(function () {
      return ordToPath(ord).then(function (path) {
        return that.$getNavTree().setSelectedPath(path);
      });
    });
  };

  /**
   * Resolve the root node to use to instantiate the component, file space and hierarchy chooser tree.
   *
   * @private
   * @returns {Promise} promise to be resolved with the component
   * instance at which to root the tree
   */
  OrdChooserWithHierarchy.prototype.$resolveTreeRootNode = function () {
    var isBajaOffline = baja.isOffline();
    var promArr = [this.getOrdBase().then(spaceUtils.resolveStation), !isBajaOffline && this.$makeHierarchySpaceNode()];
    return Promise.all(promArr).spread(function (station, hierarchySpaceNode) {
      var nodes = [new BajaComplexTreeNode(station)];
      if (hierarchySpaceNode) {
        nodes.push(hierarchySpaceNode);
      }
      return stationTree.makeSessionNode(nodes);
    });
  };
  /**
   * Overriden method to handle the selection of hierarchy node
   * @returns {*}
   */
  OrdChooserWithHierarchy.prototype.$commitSelectedNode = function () {
    var that = this,
        selectedNodes = that.$getNavTree().getSelectedNodes();
    if (selectedNodes.length === 1) {
      var node = selectedNodes[0],
          pathArray = node.getFullPath();
      if (pathArray[1] === "hierarchy") {
        // Handle the hierarchy ORD
        var ord = hierarchyOrdToOrd(pathArray);
        return that.$updateDisplay(ord, node.getIcon()).then(_.constant(ord));
      } else {
        // delegate to default functionality in OrdChooser
        return OrdChooser.prototype.$commitSelectedNode.apply(that, arguments);
      }
    }
    return Promise.resolve(null);
  };

  /**
   * Resolves the hierarchy space, then builds the tree node for it.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/BajaNavTreeNode|null>} promise
   * to be resolved with the hierarchy space node, or `null` if the station has
   * no hierarchy service installed
   */
  OrdChooserWithHierarchy.prototype.$makeHierarchySpaceNode = function () {
    return baja.Ord.make("local:|hierarchy:").get().then(function (hierarchySpace) {
      var node = new BajaNavTreeNode(hierarchySpace);
      node.toDisplay = function () {
        return Promise.resolve(lex.get("analytics.hierarchy"));
      };
      return node;
    }).catch(_.constant(null));
  };

  return OrdChooserWithHierarchy;
});
