/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * @file Custom time range field editor for analytic web chart
 * @author Upender Paravastu
 */
/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeSelector
 */
define(['baja!', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'Promise', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'bajaux/events', 'moment', 'dialogs', 'nmodule/webEditors/rc/fe/baja/AbsTimeEditor', 'nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeFE', 'nmodule/analytics/rc/report/fe/AnalyticDayOfWeekEditor', 'nmodule/analytics/rc/chart/base/analyticEvents', 'nmodule/analytics/rc/report/util/reportUtils', 'hbs!nmodule/analytics/rc/report/templates/UxAnalyticReportDateTimeSelector', 'lex!analytics', 'baja!analytics:AnalyticReportTimeRange,baja:AbsTime', 'css!nmodule/analytics/rc/report/styles/uxstyles', 'css!nmodule/js/rc/dialogs/dialogs'], function (baja, Widget, BaseEditor, Promise, $, _, fe, feDialogs, events, moment, dialogs, AbsTimeEditor, AnalyticReportTimeRangeFE, AnalyticDayOfWeekEditor, analyticEvents, reportUtils, UxAnalyticReportDateTimeSelector, lexicon, types) {

    'use strict';

    var lex = lexicon[0];
    /**
     *  <pre> AnalyticReportTimeRangeSelector </pre> is the field  editor for time range selection in
     *  analytics UX reports.
     *  It builds upon the functionality of the frozenEnum Editor and adds
     *  extra time range options with their respective handling
     * @class
     * @alias module:nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeSelector
     * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
     */
    var AnalyticReportTimeRangeSelector = function AnalyticReportTimeRangeSelector() {
        BaseEditor.apply(this, arguments);
    };

    // Inheriting AnalyticDataEditorFE from BaseEditor
    AnalyticReportTimeRangeSelector.prototype = Object.create(BaseEditor.prototype);
    // Setting the constructor
    AnalyticReportTimeRangeSelector.prototype.constructor = AnalyticReportTimeRangeSelector;

    /**
     * Initialize the time range editor
     * @param element
     */
    AnalyticReportTimeRangeSelector.prototype.doInitialize = function (dom) {
        dom.append(UxAnalyticReportDateTimeSelector({
            dateTime: lex.get("report.settings.datetime"),
            from: lex.get("report.settings.datetime.from"),
            to: lex.get("report.settings.datetime.to")
        }));
    };

    /**
     * This method takes in string format of the time range string and makes additional
     * time range info for 'from', 'current' , 'previous' and 'timeRange' options.
     * @param value
     */
    AnalyticReportTimeRangeSelector.prototype.doLoad = function (value) {
        var that = this,
            trDom = ".timerange-fe",
            jq = that.jq(),
            props = this.properties();
        return fe.buildFor({
            dom: $(trDom, jq),
            value: value,
            type: AnalyticReportTimeRangeFE
        }).then(function (trfeEd) {
            that.$trEditor = trfeEd;
            var isCustomTimeRange = that.$trEditor.timeRangeOrdinal === 3;
            var $editors = [];
            // $editors.push(trFE);
            $editors.push(fe.buildFor({
                dom: $('.start-date-value', jq),
                value: baja.$("baja:AbsTime").decodeFromString(props.getValue("stIntrnl")),
                // properties: {showTime: false},
                readonly: !isCustomTimeRange,
                formFactor: "mini"
            }));
            $editors.push(fe.buildFor({
                dom: $('.end-date-value', jq),
                value: baja.$("baja:AbsTime").decodeFromString(props.getValue("etIntrnl")),
                // properties: {showTime: false},
                readonly: !isCustomTimeRange,
                formFactor: "mini"
            }));
            return Promise.all($editors).spread(function (stDtFe, endDtFe) {
                that.$editorList = [stDtFe, endDtFe];
                that.$registerEvents();
                return Promise.resolve(that.$editorList);
            });
        });
    };

    /**
     *  Register events
     */
    AnalyticReportTimeRangeSelector.prototype.$registerEvents = function () {
        var that = this;
        that.$trEditor.jq().on(analyticEvents.TIME_RANGE_CHANGED, function (e, trEnumTag, trVal) {
            that.setModified(true);
            var isCustomTimeRange = trEnumTag.getTag() === "timeRainge",
                promiseList = [];
            _.each(that.$editorList, function (editor, index) {
                promiseList.push(editor.setReadonly(!isCustomTimeRange));
            });
            if (!isCustomTimeRange) {
                that.$updateDateTime(trVal);
            }
        });
    };

    /**
     * Update the date time string
     * @param timeRangeStr
     */
    AnalyticReportTimeRangeSelector.prototype.$updateDateTime = function (timeRangeStr) {
        var that = this;
        return reportUtils.getDatesFromTimeRange(timeRangeStr).then(function (timerange) {
            var startTime = timerange.getStartTime(),
                endTime = timerange.getEndTime();
            // If time range is selected as 'all', then the end_time is set to baja's 'now', this is only for visual purposes.
            if (timeRangeStr === "all") {
                endTime = baja.AbsTime.now();
            }
            var editors = that.$editorList,
                promiseList = [];
            var stDateEditor = editors[0],
                endDateEditor = editors[1];
            if (stDateEditor) {
                promiseList.push(stDateEditor.load(startTime));
            }
            if (endDateEditor) {
                promiseList.push(endDateEditor.load(endTime));
            }
            return promiseList;
        });
    };

    /**
     * read the value from the time range and time range info
     * @param element
     */
    AnalyticReportTimeRangeSelector.prototype.doRead = function () {
        var that = this;
        if (that.$editorList && that.$editorList.length === 2) {
            var stDtFeRead = that.$editorList[0].doRead(),
                endDtFeRead = that.$editorList[1].doRead();
            return Promise.all([stDtFeRead, endDtFeRead]).spread(function (stDtFeVal, endDtFeVal) {
                var timeRangeInfo = that.$trEditor.timeRangeInfo;
                var trInfo = timeRangeInfo.timeRainge;
                trInfo.ordinal = that.$trEditor.timeRangeOrdinal;
                // Strip the seconds and milli seconds from the start and end time.
                trInfo.startTime = reportUtils.getHourMinuteOnly(stDtFeVal);
                trInfo.endTime = reportUtils.getHourMinuteOnly(endDtFeVal);
                return Promise.resolve({
                    tag: that.$trEditor.doRead(),
                    ordinal: trInfo.ordinal
                });
            });
        }
    };
    return AnalyticReportTimeRangeSelector;
});
