/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/AnalyticDataTagEditor
 */
define(['baja!', 'lex!analytics', 'jquery', 'Promise', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/fe', 'bajaux/events', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/analytics/rc/report/templates/UxDataTagEditor', 'nmodule/analytics/rc/report/util/reportConstants', 'css!nmodule/analytics/rc/report/styles/uxstyles'], function (baja, lexs, $, Promise, Widget, fe, events, BaseEditor, UxDataTagEditor, reportConstants) {
  'use strict';

  var lex = lexs[0];

  /**
   * A field editor for selecting data tags
   * @class
   * @alias module:nmodule/analytics/rc/report/fe/AnalyticDataTagEdito
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @param params
   */
  var AnalyticDataTagEditor = function AnalyticDataTagEditor(params) {
    BaseEditor.apply(this, arguments);
  };

  AnalyticDataTagEditor.prototype = Object.create(BaseEditor.prototype);

  AnalyticDataTagEditor.prototype.constructor = AnalyticDataTagEditor;

  /**
   * Filter the content in select box
   * @param search
   */
  function filterTest(search, selectElement) {
    search = search || "";
    var regex = new RegExp(search, "gi");

    var optns = selectElement.empty().data('options');
    $.each(optns, function (i) {
      var option = optns[i];
      if (option.text.match(regex) !== null) {
        selectElement.append($('<option>').text(option.text).val(option.value));
      }
    });
  }

  /**
   * Creates the template for the data tag editor
   * Makes an RPC call to station to get the available tags
   * @param {jQuery} dom
   */
  AnalyticDataTagEditor.prototype.doInitialize = function (dom) {
    var that = this;
    var jq = that.jq();
    var filterText = lex.get("datatag.filterText");
    return baja.rpc({
      ord: "service:analytics:AnalyticService",
      args: [{ 'selNode': baja.Ord.DEFAULT }],
      methodName: "getTagDataForBox"
    }).then(function (data) {
      // Populate the DOM with the template.
      dom.html(UxDataTagEditor({
        filterText: filterText,
        tagList: data
      }));
    }).catch(function (err) {
      jq.html(UxDataTagEditor({
        filterText: filterText,
        tagList: []
      }));
    }).finally(function () {
      that.registerEvents();
      return BaseEditor.prototype.doInitialize.apply(that, arguments);
    });
  };

  /**
   * Loads the data in to the data tag editor.
   * @param {jQuery} dom
   */
  AnalyticDataTagEditor.prototype.doLoad = function (val) {
    this.$getDataElement().val(val);
    return Promise.resolve(val);
  };

  /**
   * Reads the data value from the date input element and constructs a
   * `Data` from it.
   * @returns {String} the read `Data` value
   */
  AnalyticDataTagEditor.prototype.doRead = function () {
    var val = this.$getDataElement().val();
    return val;
  };

  /**
   * Enables or disables the data input element.
   * @param {Boolean} enabled
   */
  AnalyticDataTagEditor.prototype.doEnabled = function (enabled) {
    this.$getDataElement().prop('disabled', !enabled || this.isReadonly());
    this.$getTagListElement().prop('disabled', !enabled || this.isReadonly());
    this.$getTagFilterTextElement().prop('disabled', !enabled || this.isReadonly());
  };

  /**
   * Enables or disables the data input element.
   * @param {Boolean} readonly
   */
  AnalyticDataTagEditor.prototype.doReadonly = function (readonly) {
    this.$getDataElement().prop('disabled', readonly || !this.isEnabled());
    this.$getTagListElement().prop('disabled', readonly || !this.isEnabled());
    this.$getTagFilterTextElement().prop('disabled', readonly || !this.isEnabled());
  };
  /**
   * Register all the events associated with this editor
   * @param {Boolean} readonly
   */
  AnalyticDataTagEditor.prototype.registerEvents = function () {
    this.$registerClearFilterTextEvent();
    this.$registerTagSelectionChangedEvent();
    this.$registerFilterEvent();
  };
  ///////////////////////////////// Private methods ///////////////////////////////////
  /**
   * Register the event for 'change' on 'select' control
   */
  AnalyticDataTagEditor.prototype.$registerTagSelectionChangedEvent = function () {
    var selectBox = this.$getTagListElement();
    var selectedText = this.$getDataElement();
    selectBox.on('change', function () {
      var optionSelected = $("option:selected", this);
      var value = optionSelected.val();
      selectedText.val(value);
    });
  };

  /**
   * Register the events for filtering tags in 'select' area
   */
  AnalyticDataTagEditor.prototype.$registerFilterEvent = function () {
    var that = this;
    var selectElement = that.$getTagListElement();
    var options = that.getAllTagElements();
    selectElement.data('options', options);

    var filterTextElement = that.$getTagFilterTextElement();
    filterTextElement.bind('keyup change', function () {
      var valThis = $(this).val().toLowerCase();
      filterTest($.trim(valThis), selectElement);
    });
  };

  /**
   * Register event for clearing the filter text
   */
  AnalyticDataTagEditor.prototype.$registerClearFilterTextEvent = function () {
    var textElement = this.$getTagFilterTextElement(),
        clearElement = this.$getTagFilterClearElement();
    clearElement.on('click', function () {
      textElement.val("");
      textElement.trigger("change");
    });
  };

  /**
   * Returns the Data element where the text will be stored
   * e.g n:history , hs:ahu
   * @private
   * @returns {jQuery}
   */
  AnalyticDataTagEditor.prototype.$getDataElement = function () {
    return $(".selected-tag-value", this.jq());
  };

  /**
   * Returns the 'select' element where the tags will be stored
   * @tags e.g n:history , hs:ahu
   * @private
   * @returns {jQuery}
   */
  AnalyticDataTagEditor.prototype.$getTagListElement = function () {
    return $(".tag-list", this.jq());
  };
  /**
   * Returns the 'search' element using which the tags will be filtered
   * @private
   * @returns {jQuery}
   */
  AnalyticDataTagEditor.prototype.$getTagFilterTextElement = function () {
    return $(".tag-filter-text", this.jq());
  };

  /**
   * Returns the 'search' element using which the tags will be filtered
   * @private
   * @returns {jQuery}
   */
  AnalyticDataTagEditor.prototype.$getTagFilterClearElement = function () {
    return $(".clear-filter-text", this.jq());
  };
  /**
   * Returns all the tags under the select box
   * @private
   * @returns {jQuery}
   */
  AnalyticDataTagEditor.prototype.getAllTagElements = function () {
    var optElements = $('select.tag-list > option', this.jq());
    var options = [];
    optElements.each(function () {
      options.push({ value: $(this).val(), text: $(this).text() });
    });
    return options;
  };
  return AnalyticDataTagEditor;
});
