/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/base/AnalyticReportConfigEditor
 */
define(['baja!', 'lex!analytics', 'jquery', 'Promise', 'moment', 'underscore', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'bajaux/events', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor', 'nmodule/analytics/rc/report/base/AnalyticReportConfigItemEditor', 'nmodule/analytics/rc/report/base/AnalyticReportFullConfigEditor', 'nmodule/analytics/rc/report/util/reportConstants', 'nmodule/analytics/rc/report/util/reportWidgetEvents', 'hbs!nmodule/analytics/rc/report/templates/UxReportConfigurationEditor', 'baja!analytics:AnalyticCompositeConfiguration,analytics:AnalyticReportConfiguration,analytics:AnalyticConfigurationItem,' + 'analytics:AnalyticReportTimeRange,analytics:AnalyticRollupDataTagItem', 'css!nmodule/analytics/rc/report/styles/uxstyles'], function (baja, lexs, $, Promise, moment, _, Widget, subscriberMixIn, fe, feDialogs, events, ComplexCompositeEditor, AnalyticReportConfigItemEditor, AnalyticReportFullConfigEditor, reportConstants, reportWidgetEvents, UxReportConfigurationEditor, types) {
  'use strict';

  var lex = lexs[0];
  /**
   * A field editor to be used in Analytics module for configuring all report
   * related parameters
   *
   * @class
   * @alias module:nmodule/analytics/rc/report/base/AnalyticReportConfigEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   */
  var AnalyticReportConfigEditor = function AnalyticReportConfigEditor(params) {
    ComplexCompositeEditor.call(this, $.extend({
      keyName: 'AnalyticReportConfigEditor'
    }, params));
    subscriberMixIn(this);
  };

  AnalyticReportConfigEditor.prototype = Object.create(ComplexCompositeEditor.prototype);

  AnalyticReportConfigEditor.prototype.constructor = AnalyticReportConfigEditor;

  /**
   * Get the list of slots to be displayed
   * @param value
   */
  AnalyticReportConfigEditor.prototype.getSlotFilter = function (value) {
    var that = this;
    var slotList = [];
    var jq = this.jq();
    var prop = this.properties();
    // Push the mandatory ones by default
    slotList.push({
      slot: "node", type: AnalyticReportConfigItemEditor,
      properties: {
        groupLimit: prop.getValue("groupLimit", -1),
        groupLimitMessage: prop.getValue("groupLimitMessage"),
        showColorSelector: prop.getValue("showColorSelector")
      }
    });
    slotList.push({ slot: "timeRange", type: AnalyticReportConfigItemEditor });

    var dataTagConfig = value.getDataTag();
    if (dataTagConfig.getPropertyRequired()) {
      $(".datatag-fe", jq).show();
      slotList.push({ slot: "dataTag", type: AnalyticReportConfigItemEditor });
    } else {
      $(".datatag-fe", jq).hide();
    }

    var baseLineConfig = value.getBaseline();
    if (baseLineConfig.getPropertyRequired()) {
      $(".baseline-fe", jq).show();
      slotList.push({
        slot: "baseline", type: AnalyticReportConfigItemEditor,
        properties: {
          "groups": that.getGroupList(value.getNode()),
          "timeRange": value.getTimeRange().getPropertyValue().getTimeRange()
        }
      });
    } else {
      $(".baseline-fe", jq).hide();
    }

    var colorRangeConfig = value.getColorRange();
    if (colorRangeConfig.getPropertyRequired()) {
      $(".colorrange-fe", jq).show();
      slotList.push({ slot: "colorRange", type: AnalyticReportConfigItemEditor });
    } else {
      $(".colorrange-fe", jq).hide();
    }

    if (_.isFunction(value.getChartValue)) {
      var chartValueCfg = value.getChartValue();
      if (chartValueCfg.getPropertyRequired()) {
        $(".chartValue-fe", jq).show();
        slotList.push({
          slot: "chartValue", type: AnalyticReportConfigItemEditor,
          properties: { "options": that.getChartValueProperties(value.getRollupDataTagMap()) }
        });
      } else {
        $(".chartValue-fe", jq).hide();
      }
    } else {
      $(".chartValue-fe", jq).hide();
    }

    var normalizationConfig = value.getNormalizationConfig();
    if (normalizationConfig.getPropertyRequired()) {
      $(".normalization-fe", jq).show();
      slotList.push({
        slot: "normalizationConfig", type: AnalyticReportConfigItemEditor,
        properties: {
          "groups": that.getGroupList(value.getNode())
        }
      });
    } else {
      $(".normalization-fe", jq).hide();
    }
    return slotList;
  };
  /**
   * Creates the template for Color Range Editor.
   * @param {jQuery} dom
   */
  AnalyticReportConfigEditor.prototype.doInitialize = function (dom) {
    this.$dom = dom;
    dom.html(UxReportConfigurationEditor({
      nodeLabel: lex.get("report.settings.nodeLabel"),
      dataTagLabel: lex.get("report.settings.dataTagLabel"),
      timeRangeLabel: lex.get("report.settings.reportPeriodLabel"),
      intervalLabel: lex.get("report.settings.intervalLabel"),
      colorRangeLabel: lex.get("report.settings.colorRangeLabel"),
      normalizationLabel: lex.get("report.settings.normalizationLabel"),
      baselineLabel: lex.get("report.settings.baselineLabel"),
      chartValue: lex.get("report.settings.chartValueLabel"),
      allOptions: lex.get("report.settings.allOptions"),
      submitCommand: lex.get("report.settings.submit"),
      advancedCommand: lex.get("report.settings.advanced"),
      saveCommand: lex.get("report.dashboard.save"),
      clearCommand: lex.get("report.dashboard.clear")
    }));
    this.handleRunReport();
    this.handleAllSettings();
    this.handleValueLoadEvents();
    return ComplexCompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   *  Open all the settings in the pop up
   */
  AnalyticReportConfigEditor.prototype.handleAllSettings = function () {
    var that = this;
    var allOptionsLink = $(".master-links", this.jq()).find('button');
    allOptionsLink.on("click", function (e) {
      e.stopPropagation();
      e.preventDefault();
      AnalyticReportConfigEditor.prototype.doRead.apply(that, {}).then(function () {
        feDialogs.showFor({
          title: lex.get("report.settings.dialog.title"),
          value: that.$comp,
          type: AnalyticReportFullConfigEditor
        }).then(function (componentDiff) {
          if (componentDiff) {
            componentDiff.apply(that.$comp, {});
            // If any perceivable changes re-render the view
            return AnalyticReportConfigEditor.prototype.doLoad.apply(that, [that.$comp]).then(function () {
              that.setModified(true);
              return that.doUpdateEditors();
            });
          }
        });
      });
    });
  };

  /**
   *  Handle all value load events
   *  e.g reportWidgetEvents.REPORT_AREA_VAL_LOADED
   */
  AnalyticReportConfigEditor.prototype.handleValueLoadEvents = function () {};
  /**
   * Run the report
   */
  AnalyticReportConfigEditor.prototype.handleRunReport = function () {
    var that = this;
    var btn = $(".submit-section", this.jq()).find("button");
    btn.on("click", function (e) {
      e.stopPropagation();
      e.preventDefault();
      Promise.all([that.toJSON(), that.getMetaData()]).spread(function (jsonData, metaData) {
        that.jq().trigger(reportWidgetEvents.REPORT_CONFIG_CHANGED, [jsonData, metaData]);
        btn.text(lex.get("report.settings.reSubmit"));
      });
    });
  };

  /**
   * Load the report configuration.
   * @param value
   * @returns {*}
   */
  AnalyticReportConfigEditor.prototype.doLoad = function (value) {
    var that = this;
    that.$comp = value;
    return ComplexCompositeEditor.prototype.doLoad.apply(that, arguments);
  };

  /**
   * Make a custom builder for ColorRangeEditor which plugs-in
   * the root DOM for individual sub editors.
   * @returns {*}
   */
  AnalyticReportConfigEditor.prototype.makeBuilder = function () {
    var that = this,
        builder = ComplexCompositeEditor.prototype.makeBuilder.apply(that, arguments);

    builder.getDomFor = function (key) {
      var jq = that.jq(),
          elem = jq; //$(".report-settings", jq);
      switch (key) {
        case 'node':
          return $('.node-fe-content', elem);
        case 'dataTag':
          return $('.datatag-fe-content', elem);
        case 'timeRange':
          return $('.timerange-fe-content', elem);
        case 'interval':
          return $('.interval-fe-content', elem);
        case 'baseline':
          return $('.baseline-fe-content', elem);
        case 'colorRange':
          return $('.colorrange-fe-content', elem);
        case 'normalizationConfig':
          return $('.normalization-fe-content', elem);
        case 'chartValue':
          return $('.chartValue-fe-content', elem);
        case 'hisTotEnabled':
          return $('.hisTotEnabled-fe-content', elem);
        case 'interpolationStatus':
          return $('.interpolationStatus-fe-content', elem);
        default:
          break;
      }
    };
    return builder;
  };
  /**
   * Apply the diff and return the whole component
   */
  AnalyticReportConfigEditor.prototype.doRead = function () {
    var that = this;
    return ComplexCompositeEditor.prototype.doRead.apply(this, arguments).then(function (diff) {
      diff.apply(that.$comp, {});
      return Promise.resolve(that.$comp);
    });
  };
  /**
   * Set Modified by enabling/disabling the 'Run Report' button
   * @param flag
   * @returns {*}
   */
  AnalyticReportConfigEditor.prototype.setModified = function (flag) {
    var btn = $(".submit-section", this.jq()).find("button");
    if (flag) {
      btn.prop('disabled', false);
    }
    this.doUpdateEditors();
  };

  /**
   * Update the child editors
   */
  AnalyticReportConfigEditor.prototype.doUpdateEditors = function () {
    var editors = this.getChildEditors(),
        pList = [];
    for (var i = 0; i < editors.length; i++) {
      pList.push(editors[i].doUpdate(this));
    }
    return Promise.all(pList);
  };

  /**
   * Serialize the configuration to JSON
   * @param flag
   */
  AnalyticReportConfigEditor.prototype.toJSON = function () {
    var that = this;
    return that.doRead().then(function (comp) {
      return Promise.resolve(comp);
    });
  };

  /**
   * Get metadata for the report
   */
  AnalyticReportConfigEditor.prototype.getMetaData = function () {
    var config = this.$comp,
        timeRange = config.getTimeRange(),
        propValue = timeRange.getPropertyValue(),
        metaData = propValue.getMetaData();
    return Promise.resolve({ timeRange: JSON.parse(metaData) });
  };

  /**
   * Get chart value properties based on combination map
   */
  AnalyticReportConfigEditor.prototype.getChartValueProperties = function (rollupDataTagMap) {
    var comboDataMap = rollupDataTagMap.getPropertyValue();
    var items = comboDataMap.getSlots().properties().dynamic().is("analytics:AnalyticRollupDataTagItem").toValueArray();
    var temp = [];
    _.each(items, function (item, i) {
      var rollup = item.getRollup();
      temp.push({
        tag: rollup.getTag(),
        display: rollup.getDisplayTag()
      });
    });
    return temp;
  };

  /**
   * Get the node configuration to be passed to baseline field editor
   */
  AnalyticReportConfigEditor.prototype.getGroupList = function (nodeRootProp) {
    var nodeRoot = nodeRootProp.getPropertyValue();
    var items = JSON.parse(nodeRoot);
    var temp = [];
    _.each(items, function (value, key) {
      temp.push({
        tag: value.guid || key,
        display: value.dn || key
      });
    });
    return temp;
  };
  /**
   * Return the Component as JSON value
   */
  AnalyticReportConfigEditor.prototype.getJSONFromConfig = function (config) {
    var ret = {};
    if (config) {
      ret.type = { value: "report" };
      var nodeItem = config.getNode(),
          timeRangeElem = config.getTimeRange(),
          dataTagElem = config.getDataTag(),
          trCfg = timeRangeElem.getPropertyValue();
      ret.node = { 'value': nodeItem.getPropertyValue() };
      ret.timeRange = {
        'value': {
          'tr': trCfg.getTimeRange(),
          'dow': trCfg.getDow().encodeToString()
        }
      };
      ret.dataTag = { 'value': dataTagElem.getPropertyValue() };

      var intervalElem = config.getInterval();
      if (intervalElem.getPropertyRequired()) {
        var intervalVal = intervalElem.getPropertyValue().encodeToString();
        var intervalSelected = intervalElem.getSelected() ? !_.isEmpty(intervalVal) : false;
        ret.interval = {
          'req': true,
          'value': intervalSelected ? intervalVal : "",
          'selected': intervalSelected
        };
      } else {
        ret.interval = { 'req': false };
      }

      var rollupElem = config.getRollup();
      if (rollupElem.getPropertyRequired()) {
        var rollupVal = rollupElem.getPropertyValue().encodeToString();
        var rollupValSelected = rollupElem.getSelected() ? !_.isEmpty(rollupVal) : false;
        ret.rollup = {
          'req': true,
          'value': rollupValSelected ? rollupVal : "",
          'selected': rollupValSelected
        };
      } else {
        ret.rollup = { 'req': false };
      }

      var aggregationElem = config.getAggregation();
      if (aggregationElem.getPropertyRequired()) {
        var aggregationVal = aggregationElem.getPropertyValue().encodeToString();
        var aggregationValSelected = aggregationElem.getSelected() ? !_.isEmpty(aggregationVal) : false;
        ret.aggregation = {
          'req': true,
          'value': aggregationValSelected ? aggregationVal : "",
          'selected': aggregationValSelected
        };
      } else {
        ret.rollup = { 'req': false };
      }

      var hisTotEnabledElem = config.getHisTotEnabled();
      if (hisTotEnabledElem.getPropertyRequired()) {
        ret.hisTotEnabled = { 'req': true, 'value': hisTotEnabledElem.getPropertyValue().encodeToString() };
      } else {
        ret.hisTotEnabled = { 'req': false };
      }

      var interpolationStatusElem = config.getInterpolationStatus();
      if (interpolationStatusElem.getPropertyRequired()) {
        ret.interpolationStatus = { 'req': true, 'value': interpolationStatusElem.getPropertyValue().encodeToString() };
      } else {
        ret.interpolationStatus = { 'req': false };
      }

      var legendPositionElem = config.getLegendPosition();
      if (legendPositionElem.getPropertyRequired()) {
        ret.legendPosition = { 'req': true, 'value': legendPositionElem.getPropertyValue().encodeToString() };
      } else {
        ret.legendPosition = { 'req': false };
      }

      var unitsElem = config.getUnits();
      if (unitsElem.getPropertyRequired()) {
        ret.units = { 'req': true, 'value': unitsElem.getPropertyValue().encodeToString() };
      } else {
        ret.units = { 'req': false, 'value': unitsElem.getPropertyValue().encodeToString() };
      }

      var dataFilterElem = config.getDataFilter();
      if (dataFilterElem.getPropertyRequired()) {
        ret.dataFilter = {
          'req': true,
          'value': dataFilterElem.getPropertyValue().encodeToString()
        };
      } else {
        ret.dataFilter = { 'req': false };
      }

      var baselineElem = config.getBaseline();
      if (baselineElem.getPropertyRequired()) {
        var baselineDetails = baselineElem.getPropertyValue();
        var baselineObj = {
          enabled: baselineDetails.getEnabled(),
          group: baselineDetails.getGroupName(),
          color: baselineDetails.getBaselineColor().encodeToString(),
          timeRange: baselineDetails.getTimeRange().getTag(),
          startTime: baselineDetails.getStartTime().encodeToString(),
          endTime: baselineDetails.getEndTime().encodeToString(),
          endTimeEnabled: baselineDetails.getEndTimeEnabled(),
          alignDOW: baselineDetails.getAlignDOW()
        };
        ret.baseline = { 'req': true, 'value': baselineObj };
      } else {
        ret.baseline = { 'req': false };
      }

      var colorRangeElem = config.getColorRange();
      if (colorRangeElem.getPropertyRequired()) {
        var colorRange = colorRangeElem.getPropertyValue();
        var obj = {
          minColor: colorRange.getMinValueColor().encodeToString(),
          avgColor: colorRange.getAvgValueColor().encodeToString(),
          maxColor: colorRange.getMaxValueColor().encodeToString()
        };
        ret.colorRange = { 'req': true, 'value': obj };
      } else {
        ret.colorRange = { 'req': false };
      }

      var mdConfig = config.getMissingDataStrategy();
      if (mdConfig.getPropertyRequired()) {
        var mdc = mdConfig.getPropertyValue();
        var objMdc = {
          enabled: mdc.getEnabled(),
          aggStrategy: mdc.getAggregationStrategy().getTag(),
          intpAlgorithm: mdc.getInterpolationAlgorithm().getTag(),
          knnValue: mdc.getKValue()
        };
        ret.mdConfig = { 'req': true, 'value': objMdc };
      } else {
        ret.mdConfig = { 'req': false };
      }
      // Below 2 methods exists only for multi binding report scenarios
      if (config.getRollupDataTagMap) {
        var comboDataMapElem = config.getRollupDataTagMap();
        if (comboDataMapElem.getPropertyRequired()) {
          var comboDataMap = comboDataMapElem.getPropertyValue();
          var items = comboDataMap.getSlots().properties().dynamic().is("analytics:AnalyticRollupDataTagItem").toValueArray();
          var temp = {};
          _.each(items, function (item, i) {
            var rollup = item.getRollup().getTag();
            var unit = item.getUnits();
            if (!temp.hasOwnProperty(rollup)) {
              temp[rollup] = [];
            }
            temp[rollup].push({ tag: item.getDataTag(), unit: unit.encodeToString() });
          });
          ret.combinationMap = { 'req': true, 'value': temp };
        } else {
          ret.combinationMap = { 'req': false };
        }
      }
      if (config.getChartValue) {
        var chartValueItem = config.getChartValue();
        if (chartValueItem.getPropertyRequired()) {
          var chartValue = chartValueItem.getPropertyValue().encodeToString();
          ret.chartValue = { 'req': true, value: chartValue };
        } else {
          ret.chartValue = { 'req': false };
        }
      }

      var normalizationElem = config.getNormalizationConfig();
      if (normalizationElem.getPropertyRequired()) {
        var normalization = normalizationElem.getPropertyValue();
        var areaNorm = normalization.getAreaNormalization(),
            degreeNorm = normalization.getDegreeDayNormalization(),
            degreeDayMap = degreeNorm.getGroupMapping(),
            areaNormMap = areaNorm.getGroupMapping();
        var normMappingItemType = "analytics:AnalyticDegreeDayNodeItem";
        var degDayItems = degreeDayMap.getSlots().properties().dynamic().is(normMappingItemType).toValueArray();
        var areaNormItems = areaNormMap.getSlots().properties().dynamic().is(normMappingItemType).toValueArray();
        var temp1 = [],
            temp2 = [];
        _.each(degDayItems, function (item, i) {
          var gn = baja.SlotPath.unescape(item.getGroupName());
          var ord = item.getDegDayRoot();
          temp1.push({ gn: baja.SlotPath.escape(gn), ord: ord.encodeToString(), guid: item.getGuid() });
        });
        _.each(areaNormItems, function (item, i) {
          var gn = baja.SlotPath.unescape(item.getGroupName());
          var ord = item.getDegDayRoot();
          temp2.push({ gn: baja.SlotPath.escape(gn), ord: ord.encodeToString(), guid: item.getGuid() });
        });
        var degDayCfg = {
          type: 'degreeday',
          selected: degreeNorm.getSelected(),
          value: degreeNorm.getValue(),
          unit: degreeNorm.getValueUnits().getOrdinal(),
          ddtype: degreeNorm.getDdtype().getOrdinal(),
          mapping: temp1
        };
        ret.normalization = {
          'req': true, 'value': [{
            type: 'area',
            selected: areaNorm.getSelected(),
            value: null,
            mapping: temp2
          }, degDayCfg]
        };
      } else {
        ret.normalization = { 'req': false };
      }
    }
    return ret;
  };

  /**
   * Show the Error associated with 'Update Report' action.
   * @param errCode
   */
  AnalyticReportConfigEditor.prototype.showError = function (errCode) {
    var errDiv = this.$getErrorDiv();
    if (errCode === 1) {
      errDiv.html(lex.get("report.update.permission.err"));
    } else {
      errDiv.html("report.update.default.err");
    }
  };

  AnalyticReportConfigEditor.prototype.$getErrorDiv = function () {
    return $(".errMessage", this.jq());
  };

  return AnalyticReportConfigEditor;
});
