/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Upender Paravastu & Sai Komaravolu
 * @file This file represents the settings window for the Web analytic charts
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/chart/fe/AnalyticUxWebChartSettingsFE
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'moment', 'underscore', 'dialogs', 'bajaux/events', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/analytics/rc/chart/fe/AnalyticDataEditorFE', 'nmodule/analytics/rc/chart/fe/AnalyticOptionalValueFE', 'hbs!nmodule/analytics/rc/chart/templates/fe/analyticWebChartSettings', 'nmodule/webEditors/rc/fe/baja/DaysOfWeekBitsEditor', 'nmodule/analytics/rc/report/fe/AnalyticDayOfWeekEditor', 'nmodule/webEditors/rc/fe/baja/BooleanEditor', 'nmodule/analytics/rc/report/fe/MissingDataStrategyEditor', 'nmodule/analytics/rc/report/fe/ReportOptionalValueFE', 'nmodule/analytics/rc/report/fe/ReportCombinerOptionalValueFE', 'lex!baja,analytics', 'baja!webEditors:FrozenEnumEditor,webEditors:UnitEditor,webEditors:StringEditor,analytics:AnalyticDataEditor', 'nmodule/webChart/rc/fe/color/ColorEditor'], function (baja, Widget, subscriberMixIn, Promise, $, moment, _, dialogs, events, fe, feDialogs, BaseEditor, AnalyticDataEditor, AnalyticOptionalValueFE, analyticWebChartSettings, DaysOfWeekBitsEditor, AnalyticDayOfWeekEditor, BooleanEditor, MissingDataStrategyEditor, ReportOptionalValueFE, ReportCombinerOptionalValueFE, lexicon, types, ColorEditor) {
  'use strict';

  var lex = lexicon[1];
  var knnVal = '';
  /**
   * A field editor for working with Settings of chart widget.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/analytics/rc/chart/fe/AnalyticUxWebChartSettingsFE
   */
  var AnalyticUxWebChartSettingsFE = function AnalyticUxWebChartSettingsFE() {
    BaseEditor.apply(this, arguments);
  };
  // Inheriting AnalyticUxWebChartSettingsFE from BaseEditor
  AnalyticUxWebChartSettingsFE.prototype = Object.create(BaseEditor.prototype);
  //Setting the constructor
  AnalyticUxWebChartSettingsFE.prototype.constructor = AnalyticUxWebChartSettingsFE;

  /**
   * Creates the necessary setting fields for configuring analytic web chart
   * @param {jQuery} dom
   */
  AnalyticUxWebChartSettingsFE.prototype.doInitialize = function (dom) {
    var that = this;
    // Load all properties in to the template
    dom.html(analyticWebChartSettings(that.initParams()));
  };

  AnalyticUxWebChartSettingsFE.prototype.getHbs = function () {
    return analyticWebChartSettings;
  };

  AnalyticUxWebChartSettingsFE.prototype.initParams = function () {
    return {
      interval: lex.get("interval"),
      rollup: lex.get("rollup"),
      aggregation: lex.get("aggregation"),
      unit: lex.get("unit"),
      data: lex.get("data"),
      dataFilter: lex.get("dataFilter"),
      brush: lex.get("brush"),
      seriesName: lex.get("seriesName"),
      daysToExclude: lex.get("daysToExclude"),
      hisTotEnabled: lex.get("hisTotEnabled"),
      mdhStrategy: lex.get("missingDataStrategy")
    };
  };

  /**
   * Loads a `{analytics.AnalyticUxWebChartParams` into web UI.
   * format.
   *
   * @param {baja.Component} settingsObj an `analytics:AnalyticUxWebChartParams`
   * @returns {Promise} promise to be resolved if the value is loaded,
   * or rejected if the `analytics.AnalyticUxWebChartParams` could not be encoded to its type
   */
  AnalyticUxWebChartSettingsFE.prototype.doLoad = function (settingsObj) {
    var that = this;
    that.settingsObj = settingsObj;
    that.jq().data("widget", that);
    // Units
    knnVal = settingsObj.getMissingDataStrategy().getKValue();
    var jq = that.jq(),
        unitText = jq.find('.unit-text'),
        unitCheckBox = jq.find(".unit-checkbox");
    if (!settingsObj.getUnit() || settingsObj.getUnit() === baja.Unit.NULL) {
      unitCheckBox.attr("checked", false);
      unitText.attr("value", "");
      unitText.attr("disabled", true);
    } else {
      unitCheckBox.attr("checked", true);
      unitText.attr("value", settingsObj.getUnit().getSymbol());
      unitText.attr("disabled", false);
    }

    unitText.on('click', function (obj) {
      var isChecked = jq.find(".unit-checkbox").is(":checked");
      if (isChecked) {
        dialogs.showOkCancel({
          title: lex.get("title.unitSelector"),
          content: function content(dlg, jq) {
            jq.html("<div class='unitsDiv'></div>");
            return new Promise(function (resolve, reject) {
              resolve(fe.buildFor({
                dom: $(".unitsDiv", jq),
                value: settingsObj.getUnit(),
                formFactor: 'mini'
              }).then(function (unitEditor) {
                // Time to register events.
                unitEditor.jq().on(events.MODIFY_EVENT, function () {
                  unitEditor.doRead().then(function (unit) {
                    settingsObj.setUnit(unit);
                  });
                });
              }));
            });
          }
        }).ok(function (obj) {
          unitText.val(settingsObj.getUnit().getSymbol());
        }).cancel(function (obj) {});
      }
    });
    unitCheckBox.on('change', function (obj) {
      var isChecked = unitCheckBox.is(":checked");
      if (isChecked) {
        unitText.attr("disabled", false);
      } else {
        unitText.attr("disabled", true);
        unitText.attr("value", "");
        settingsObj.setUnit(baja.Unit.NULL);
      }
    });
    // Load the field editors for each of the properties.
    return that.buildFieldEditors(settingsObj);
  };

  /**
   * Reads the  value from the settings user interface and constructs a
   * `analytics.AnalyticUxWebChartParams` from it.
   *
   * @returns {{analytics.AnalyticUxWebChartParams} the read `{analytics.AnalyticUxWebChartParams` value
   * @throws {Error} if the input contains an invalid date value.
   */
  AnalyticUxWebChartSettingsFE.prototype.doRead = function () {
    var that = this;
    var promArray = [],
        optionalParam;
    _.each(that.getChildEditors(), function (ce) {
      promArray.push(ce.read().then(function (val) {
        if (ce.$keyName === "interval") {
          optionalParam = baja.$("analytics:OptionalParam");
          optionalParam.setValue(baja.$("analytics:Interval").make(val.value));
          optionalParam.setSelected(val.isSelected);
          that.settingsObj.setInterval(optionalParam);
        } else if (ce.$keyName === "aggregation") {
          optionalParam = baja.$("analytics:OptionalParam");
          optionalParam.setValue(baja.$("analytics:Combination").make(val.value));
          optionalParam.setSelected(val.isSelected);
          that.settingsObj.setAggregation(optionalParam);
        } else if (ce.$keyName === "rollup") {
          optionalParam = baja.$("analytics:OptionalParam");
          optionalParam.setValue(baja.$("analytics:Combination").make(val.value));
          optionalParam.setSelected(val.isSelected);
          that.settingsObj.setRollup(optionalParam);
        } else {
          that.settingsObj["set" + ce.$keyName.capitalizeFirstLetter()](val);
        }
        return Promise.resolve(true);
      }));
    });
    return Promise.all(promArray).then(function (vals) {
      return that.settingsObj;
    });
  };
  /**
   * save callback handler
   * @param value
   * @param params
   */
  AnalyticUxWebChartSettingsFE.prototype.doSave = function (value, params) {};
  /**
   * Enables or disables the settings UI.
   *
   * @param {Boolean} enabled
   */
  AnalyticUxWebChartSettingsFE.prototype.doEnabled = function (enabled) {
    this.$getDataElement().prop('disabled', !enabled || this.isReadonly());
  };

  /**
   * Enables or disables the settings UI.
   * @param {Boolean} readonly
   */
  AnalyticUxWebChartSettingsFE.prototype.doReadonly = function (readonly) {
    this.$getDataElement().prop('disabled', readonly || !this.isEnabled());
  };

  AnalyticUxWebChartSettingsFE.prototype.$getDataElement = function () {
    return this.jq();
  };

  /**
   * Build Field Editors
   * @param settingsObj
   * @returns {Array Promise}
   */
  AnalyticUxWebChartSettingsFE.prototype.initFieldEditors = function (settingsObj) {
    var that = this,
        fes = [],
        msg = lex.get("testTotalizeNeql.message");
    fes.push({
      dom: $(".intervalFE", that.jq()),
      value: { value: settingsObj.getInterval().getValue(), isSelected: settingsObj.getInterval().getSelected() },
      type: AnalyticOptionalValueFE,
      formFactor: 'mini',
      keyName: "interval"
    }, {
      dom: $(".rollupFE", that.jq()),
      value: { value: settingsObj.getRollup().getValue(), isSelected: settingsObj.getRollup().getSelected() },
      type: AnalyticOptionalValueFE,
      formFactor: 'mini',
      keyName: "rollup"
    }, {
      dom: $(".aggregationFE", that.jq()),
      value: { value: settingsObj.getAggregation().getValue(), isSelected: settingsObj.getAggregation().getSelected() },
      type: AnalyticOptionalValueFE,
      formFactor: 'mini',
      keyName: "aggregation"
    }, {
      dom: $(".dataFE", that.jq()),
      type: AnalyticDataEditor,
      value: settingsObj.getData(),
      formFactor: 'mini',
      keyName: "data"
    }, {
      dom: $(".dataFilterFE", that.jq()),
      value: settingsObj.getDataFilter(),
      formFactor: 'mini',
      keyName: "dataFilter"
    }, {
      dom: $(".brush", that.jq()),
      type: ColorEditor,
      value: settingsObj.getBrush(),
      formFactor: 'mini',
      keyName: "brush"
    }, {
      dom: $(".seriesName", that.jq()),
      value: settingsObj.getSeriesNameBFormat(),
      formFactor: 'mini',
      keyName: "seriesNameBFormat"
    }, {
      dom: $(".daysToExcludeFE", that.jq()),
      type: AnalyticDayOfWeekEditor,
      value: settingsObj.getDaysToExclude(),
      keyName: "daysToExclude"
    }, {
      dom: $(".hisTotEnabledFE", that.jq()),
      type: BooleanEditor,
      value: settingsObj.getHisTotEnabled(),
      properties: {
        trueText: msg, falseText: msg
      },
      keyName: "hisTotEnabled"
    }, {
      dom: $(".mdhStrategyFE", that.jq()),
      type: MissingDataStrategyEditor,
      value: settingsObj.getMissingDataStrategy(),
      keyName: "missingDataStrategy"
    });
    return fes;
  };

  AnalyticUxWebChartSettingsFE.prototype.buildFieldEditors = function (settingsObj) {
    var that = this;
    var feArr = that.initFieldEditors(settingsObj);
    var promArray = [];
    _.each(feArr, function (feObj) {
      promArray.push(fe.buildFor(feObj));
    });
    return Promise.all(promArray).then(function () {
      applyListeners(that, settingsObj);
    });
  };

  function applyListeners(that, settingsObj) {
    var rollupCheckBox = that.jq().find('.rollupFE input[type=checkbox]'),
        rollupText = that.jq().find('.rollupFE .fe-text'),
        aggCheckBox = that.jq().find('.aggregationFE input[type=checkbox]'),
        aggText = that.jq().find('.aggregationFE .fe-text'),
        intervalCheckBox = that.jq().find('.intervalFE input[type=checkbox]'),
        intervalText = that.jq().find('.intervalFE .fe-text'),
        knnValue = that.jq().find('.mdhStrategyFE input[type=Number]');
    if (!settingsObj.getRollup().getSelected()) {
      rollupText.val('');
    }
    if (!settingsObj.getAggregation().getSelected()) {
      aggText.val('');
    }
    if (!settingsObj.getInterval().getSelected()) {
      intervalText.val('');
    }
    rollupCheckBox.change(function () {
      if (rollupCheckBox.is(":checked")) {
        rollupText.val(settingsObj.getRollup().getValue().getTag().capitalizeFirstLetter());
      } else {
        rollupText.val('');
      }
    });
    aggCheckBox.change(function () {
      if (aggCheckBox.is(":checked")) {
        aggText.val(settingsObj.getAggregation().getValue().getTag().capitalizeFirstLetter());
      } else {
        aggText.val('');
      }
    });
    intervalCheckBox.change(function () {
      if (intervalCheckBox.is(":checked")) {
        intervalText.val(settingsObj.getInterval().getValue().getTag().capitalizeFirstLetter());
      } else {
        intervalText.val('');
      }
    });
    knnValue.change(function () {
      if (isFinite(knnValue[0].value)) {
        if (knnValue[0].value < 1 || knnValue[0].value > 30) {
          if (knnValue[0].value > 30) {
            knnValue[0].value = 30;
          } else {
            knnValue[0].value = knnVal;
          }
          return dialogs.showOk({
            title: lex.getSafe("error.title"),
            content: lex.getSafe("analytics.mdc.min.max.error")
          });
        }
      } else {
        knnValue[0].value = 1;
      }
    });
  }

  return AnalyticUxWebChartSettingsFE;
});
