/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * @file Custom time range field editor for analytic web chart
 * @author Upender Paravastu & Sai Komaravolu
 */
define(['baja!', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'Promise', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/analytics/rc/util/analyticsUtil', 'bajaux/events', 'moment', 'dialogs', 'nmodule/analytics/rc/chart/base/analyticEvents', 'hbs!nmodule/analytics/rc/chart/templates/base/timeRange_from_tmpl', 'hbs!nmodule/analytics/rc/chart/templates/base/timeRange_previous_tmpl', 'nmodule/analytics/rc/chart/fe/StartEndTimeRangeEditor', 'lex!analytics', "baja!analytics:UxAnalyticTimeRangeType," + "analytics:TimeRangeStringLabel,analytics:AnalyticChartTimeRange", 'css!nmodule/analytics/rc/chart/styles/ChartStyles', 'css!nmodule/js/rc/dialogs/dialogs'], function (baja, Widget, BaseEditor, Promise, $, _, fe, feDialogs, analyticsUtil, events, moment, dialogs, analyticEvents, fromTemplate, previousTemplate, StartEndTimeRangeEditor, lexicon, types) {

    'use strict';
    /**
     *  <pre> AnalyticTimeRangeFE </pre> is the field  editor for time range selection in
     *  analytics UX charts.
     *  It builds upon the functionality of the frozenEnum Editor and adds
     *  extra time range options with their respective handling
     * @constructor
     */

    var AnalyticTimeRangeFE = function AnalyticTimeRangeFE() {
        BaseEditor.apply(this, arguments);
    };
    // Inheriting AnalyticDataEditorFE from BaseEditor
    AnalyticTimeRangeFE.prototype = Object.create(BaseEditor.prototype);
    // Setting the constructor
    AnalyticTimeRangeFE.prototype.constructor = AnalyticTimeRangeFE;

    var lex = lexicon[0];
    /**
     * Initialize the time range editor
     * @param element
     */
    AnalyticTimeRangeFE.prototype.doInitialize = function (element) {
        this.timeRangeOrdinal = 0;
        this.timeRangeInfo = { "from": {}, "current": {}, "next": {}, "previous": {}, "timeRainge": {} };
    };
    /**
     * read the value from the time range and time range info
     * @param element
     */
    AnalyticTimeRangeFE.prototype.doRead = function () {
        return this.encodeToString();
    };
    /**
     * This method takes in string format of the time range string and makes additional
     * time range info for 'from', 'current' , 'previous' and 'timeRange' options.
     * @param value
     */
    AnalyticTimeRangeFE.prototype.doLoad = function (value) {
        // decode the value and populate timeRangeInfo if applicable
        if (value) {
            value = decodeURIComponent(value);
            this.decodeFromString(value);
        }
        // The values are now decoded. Time to build the field editors

        var that = this,
            prevTimeRange;
        // Initiate the time range tab using Frozen enum editor
        // Time Interval
        var jq = that.jq();
        if (jq.find(".timeRangeElemFE").length === 0) {
            jq.append("<div class='timeRangeElemFE'></div>");
        }
        fe.buildFor({
            dom: $(".timeRangeElemFE", jq),
            properties: { uxFieldEditor: 'webEditors:FrozenEnumEditor' },
            value: baja.$("analytics:UxAnalyticTimeRangeType").make(that.timeRangeOrdinal)
        }).then(function (timeRangeEditor) {
            // Now handle the event registration
            prevTimeRange = prevTimeRange || timeRangeEditor.doRead();
            timeRangeEditor.jq().off(events.MODIFY_EVENT);
            timeRangeEditor.jq().on(events.MODIFY_EVENT, function (event, fEnum) {
                event.stopPropagation();
                // that.setModified(true);
                var enumVal = fEnum.doRead();
                var ordinal = enumVal.getOrdinal();
                that.timeRangeOrdinal = ordinal;
                // Do special handling for "from",  "timeRange", "current" and "previous"
                // In other cases just be quiet
                if (ordinal === 0) // From Template handling for time range
                    {
                        dialogs.showOkCancel({
                            title: lex.get("analytics.timerange." + enumVal.getTag()),
                            content: function content(dlg, jq) {
                                jq.html(fromTemplate($.extend({
                                    fromLbl: lex.get("time.from"),
                                    toLbl: lex.get("time.to")
                                }, that.getFromOptions())));
                                that.setValuesToDom(jq, enumVal);
                                jq.find(".fromOffsetEnabled").on("change", function (obj) {
                                    var fromOffsetEnabled = $(this).is(':checked');
                                    if (!fromOffsetEnabled) {
                                        jq.find('.fromOperation, .fromStepFactor, .fromCount, .fromOffset').attr("disabled", true).toggleClass('ux-disabled', true);
                                    } else {
                                        jq.find('.fromOperation, .fromStepFactor, .fromCount, .fromOffset').toggleClass('ux-disabled', false).attr("disabled", false);
                                    }
                                });
                                jq.find(".toOperation").on("change", function (obj) {
                                    if ($(this).val() === "Now") {
                                        jq.find('.toCount, .toOffset, .toStepFactor').attr("disabled", true).toggleClass("ux-disabled", true);
                                    } else {
                                        jq.find('.toCount, .toOffset, .toStepFactor').attr("disabled", false).toggleClass("ux-disabled", false);
                                    }
                                });
                                return jq;
                            }
                        }).ok(function (obj) {
                            // Populate the time range info
                            prevTimeRange = timeRangeEditor.doRead();
                            var dom = obj.jq();
                            var trInfo = that.timeRangeInfo.from;
                            trInfo.ordinal = that.timeRangeOrdinal;
                            trInfo.fromAlignment = dom.find(".fromAlignment").val(); // This will be there
                            var fromOffsetEnabled = dom.find(".fromOffsetEnabled").is(":checked");
                            trInfo.fromOffsetEnabled = fromOffsetEnabled;
                            if (fromOffsetEnabled) {
                                var fromOperation = dom.find(".fromOperation").val() === "-" ? "-" : "";
                                var fromCount = dom.find(".fromOffset").val();
                                var fromStep = dom.find(".fromStepFactor").val();
                                if (fromStep === "P{0}W") {
                                    fromStep = "P{0}D";
                                    fromCount *= 7;
                                }
                                trInfo.fromOperation = fromOperation;
                                trInfo.fromOffset = fromOperation + fromStep.replace("{0}", fromCount);
                            }
                            var toOperation = $(".toOperation", dom).val();
                            toOperation = toOperation === "-" ? "-" : "";
                            var toCount = $(".toOffset", dom).val();
                            var toStep = $(".toStepFactor", dom).val();
                            if (toStep === "Now") {
                                trInfo.toOffset = "Now";
                            } else {
                                trInfo.toOffset = toOperation + toStep.replace("{0}", toCount);
                            }
                            that.jq().trigger(analyticEvents.TIME_RANGE_CHANGED, [timeRangeEditor.doRead(), that.doRead()]);
                        }).cancel(function (obj) {
                            return timeRangeEditor.load(prevTimeRange);
                        });
                    } else if (ordinal === 1 || ordinal === 2) {
                    //"current" or "previous"
                    dialogs.showOkCancel({
                        title: lex.get("analytics.timerange." + enumVal.getTag()),
                        content: function content(dlg, jq) {
                            jq.html(previousTemplate(that.getPCNOptions()));
                            that.setValuesToDom(jq, enumVal);
                            return jq;
                        }
                    }).ok(function (obj) {
                        prevTimeRange = timeRangeEditor.doRead();
                        var dom = obj.jq();
                        var trInfo = that.timeRangeInfo[enumVal.getTag()];
                        var val = dom.find(".fromCountText").val();
                        trInfo.ordinal = that.timeRangeOrdinal;
                        trInfo.val = parseInt(val);
                        trInfo.duration = dom.find(".fromStepFactor").val();
                        // Trigger a modify and let subscriber do the rest.
                        that.jq().trigger(analyticEvents.TIME_RANGE_CHANGED, [timeRangeEditor.doRead(), that.doRead()]);
                    }).cancel(function (obj) {
                        return timeRangeEditor.load(prevTimeRange);
                    });
                } else if (ordinal === 3) {
                    // Custom time range
                    that.handleCustomTimeRange(timeRangeEditor, enumVal, prevTimeRange);
                } else {
                    prevTimeRange = timeRangeEditor.doRead();
                    that.jq().trigger(analyticEvents.TIME_RANGE_CHANGED, [timeRangeEditor.doRead(), that.doRead()]);
                }
            });
        });
        //////////////////////////////////////////////////////////////
    };

    /**
     * Temporary fix for the custom time range "date Time" option
     */
    AnalyticTimeRangeFE.prototype.handleCustomTimeRange = function (timeRangeEditor, currentTimeRangeEnum, prevTimeRangeEnum) {
        var that = this;
        var sTime = !_.isEmpty(that.timeRangeInfo.timeRainge) ? that.timeRangeInfo.timeRainge.startTime : analyticsUtil.getAbsTime(moment().subtract(1, 'days'));
        var eTime = !_.isEmpty(that.timeRangeInfo.timeRainge) ? that.timeRangeInfo.timeRainge.endTime : baja.AbsTime.now();
        var chartTimeRange = baja.$('analytics:AnalyticChartTimeRange', {
            startTime: sTime,
            endTime: eTime,
            start: baja.$('analytics:TimeRangeStringLabel', { timeRangeLabel: lex.get("time.start") }),
            end: baja.$('analytics:TimeRangeStringLabel', { timeRangeLabel: lex.get("time.end") })
        });
        feDialogs.showFor({
            title: lex.get("analytics.timerange." + currentTimeRangeEnum.getTag()),
            value: chartTimeRange,
            type: StartEndTimeRangeEditor
        }).then(function (chartTimeRangeDiff) {
            if (chartTimeRangeDiff) {
                chartTimeRangeDiff.apply(chartTimeRange);
                var trInfo = that.timeRangeInfo.timeRainge;
                trInfo.ordinal = that.timeRangeOrdinal;
                var startTime = chartTimeRange.getStartTime(),
                    endTime = chartTimeRange.getEndTime();
                trInfo.startTime = startTime ? startTime : trInfo.startTime;
                trInfo.endTime = endTime ? endTime : trInfo.endTime;
                that.jq().trigger(analyticEvents.TIME_RANGE_CHANGED, [timeRangeEditor.doRead(), that.doRead()]);
            } else {
                return timeRangeEditor.load(prevTimeRangeEnum);
            }
        });
    };

    /**
     * Set default values for 'from', 'to' , 'previous' options in UI
     * For time being let them take the default values template gives them !!!
     */
    AnalyticTimeRangeFE.prototype.setValuesToDom = function (jq, enumVal) {
        var that = this;
        var ordinal = enumVal.getOrdinal(),
            factor;
        var timeRangeInfo = that.timeRangeInfo,
            timeRange;
        var fromOptions = that.getFromOptions();
        if (ordinal === 0) {
            // Populate From DOM
            if (!_.isEmpty(timeRangeInfo.from)) {
                timeRange = analyticsUtil.extractTimeRangeForFrom(timeRangeInfo.from.fromOffset, timeRangeInfo.from.toOffset);
                jq.find(".fromAlignment").val(timeRangeInfo.from.fromAlignment);
                jq.find(".fromOffsetEnabled").attr("checked", timeRange.isMainFactorSelected);
                if (timeRange.isMainFactorSelected) {
                    jq.find('.fromOperation, .fromStepFactor, .fromCount, .fromCountText').attr("disabled", false).toggleClass('ux-disabled', false);
                    jq.find(".fromOperation").val(timeRange.fromOperation || "+");
                    factor = _.find(fromOptions.from.stepFactor, function (factor) {
                        if (factor.val === timeRange.fromStepFactor) {
                            return true;
                        }
                    });
                    jq.find(".fromStepFactor").val(factor.key);
                    jq.find(".fromCount").val(timeRange.fromCount);
                    jq.find(".fromCountText").attr("value", timeRange.fromCount);
                } else {
                    jq.find('.fromOperation, .fromStepFactor, .fromCount, .fromCountText').attr("disabled", true).toggleClass('ux-disabled', true);
                }

                jq.find(".toOperation").attr("disabled", false).val(timeRange.toOperation).toggleClass('ux-disabled', false);

                if (timeRange.toOperation === "Now") {
                    jq.find('.toCount, .toCountText, .toStepFactor').attr("disabled", true).toggleClass('ux-disabled', true);
                } else {
                    jq.find('.toCount, .toCountText, .toStepFactor').attr("disabled", false).toggleClass('ux-disabled', false);
                    jq.find(".toCount").val(timeRange.toCount);
                    jq.find(".toCountText").attr("value", timeRange.toCount);
                    factor = _.find(fromOptions.to.stepFactor, function (factor) {
                        if (factor.val === timeRange.toStepFactor) {
                            return true;
                        }
                    });
                    jq.find(".toStepFactor").val(factor.key);
                }
            } else {
                jq.find('.fromOperation, .fromStepFactor, .fromCount, .fromCountText, .toCount, .toCountText, .toStepFactor, .toOperation').attr("disabled", false).toggleClass('ux-disabled', false);
                // Set it to default.
                jq.find(".fromAlignment").val(fromOptions.mainFactor[0].key);
                jq.find(".fromOffsetEnabled").attr("checked", fromOptions.isMainFactorSelected);
                jq.find(".fromOperation").val(fromOptions.from.operation[0].val);
                jq.find(".fromStepFactor").val(fromOptions.from.stepFactor[0].key);
                jq.find(".fromCount").val(fromOptions.from.count[0]);
                jq.find(".fromCountText").attr("value", fromOptions.from.count[0]);
                jq.find(".toCount").val(fromOptions.to.count[0]);
                jq.find(".toCountText").attr("value", fromOptions.to.count[0]);
                jq.find(".toStepFactor").val(fromOptions.to.stepFactor[0].key);
                jq.find(".toOperation").val(fromOptions.to.operation[0].key);
            }
        } else if (ordinal === 1 || ordinal === 2) {
            // Curremt amd Previous
            fromOptions = that.getPCNOptions();
            jq.find('.fromStepFactor, .fromCount, .fromCountText').attr("disabled", false).toggleClass('ux-disabled', false);
            var cOrP;
            if (!_.isEmpty(timeRangeInfo.current) && ordinal === 1) {
                cOrP = timeRangeInfo.current;
            } else if (!_.isEmpty(timeRangeInfo.previous) && ordinal === 2) {
                cOrP = timeRangeInfo.previous;
            }
            if (!_.isEmpty(cOrP)) {
                jq.find(".fromStepFactor").val(cOrP.duration);
                jq.find(".fromCount").val(cOrP.val);
                jq.find(".fromCountText").attr("value", cOrP.val);
            } else {
                jq.find(".fromStepFactor").val(fromOptions.from.stepFactor[0].key);
                jq.find(".fromCount").val(fromOptions.from.count[0]);
                jq.find(".fromCountText").attr("value", fromOptions.from.count[0]);
            }
        }
    };
    /**
     * Template object used as input for HBS templates
     * @returns {{mainFactor: string[], isMainFactorSelected: boolean, from: {operation: string[], count: number[], stepFactor: string[]}, to: {operation: string[], count: number[], stepFactor: string[]}}}
     * @constructor
     */
    AnalyticTimeRangeFE.prototype.getFromOptions = function () {
        var counts = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 15, 20, 24, 28, 30, 36, 45, 48, 60, 72, 90, 96, 180, 365];
        var steps = [{ key: 'PT{0}M', val: lex.get("time.step.minutes") }, {
            key: 'PT{0}H',
            val: lex.get("time.step.hours")
        }, { key: 'P{0}D', val: lex.get("time.step.days") }, { key: 'P{0}W', val: lex.get("time.step.weeks") }, { key: 'P{0}M', val: lex.get("time.step.months") }, { key: 'PT{0}Y', val: lex.get("time.step.years") }];
        return {
            mainFactor: [{ key: 'Minute', val: lex.get("time.from.minute") }, {
                key: 'Hour',
                val: lex.get("time.from.hour")
            }, { key: 'Day', val: lex.get("time.today") }, { key: 'Week', val: lex.get("time.from.week") }, { key: 'Month', val: lex.get("time.from.month") }, { key: 'Year', val: lex.get("time.from.year") }],
            isMainFactorSelected: true,
            defaultCount: 1,
            from: {
                operation: [{ key: "+", val: lex.get("time.operation.add") }, {
                    key: "-",
                    val: lex.get("time.operation.minus")
                }],
                count: counts,
                stepFactor: steps
            },
            to: {
                operation: [{ key: "+", val: lex.get("time.operation.add") }, {
                    key: "-",
                    val: lex.get("time.operation.minus")
                }, { key: 'Now', val: lex.get("time.operation.now") }],
                count: counts,
                stepFactor: steps
            }
        };
    };
    /**
     * Template object used as input for HBS templates
     * @returns {{mainFactor: string[], isMainFactorSelected: boolean, from: {operation: string[], count: number[], stepFactor: string[]}, to: {operation: string[], count: number[], stepFactor: string[]}}}
     * @constructor
     */
    AnalyticTimeRangeFE.prototype.getPCNOptions = function () {
        var counts = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 15, 20, 24, 28, 30, 36, 45, 48, 60, 72, 90, 96, 180, 365];
        var steps = [{ key: 'minutes', val: lex.get("time.step.minutes") }, {
            key: 'hours',
            val: lex.get("time.step.hours")
        }, { key: 'days', val: lex.get("time.step.days") }, { key: 'weeks', val: lex.get("time.step.weeks") }, { key: 'months', val: lex.get("time.step.months") }, { key: 'years', val: lex.get("time.step.years") }];
        return {
            defaultCount: 1,
            from: {
                operation: [{ key: "+", val: lex.get("time.operation.add") }, {
                    key: "-",
                    val: lex.get("time.operation.minus")
                }],
                count: counts,
                stepFactor: steps
            }
        };
    };
    /**
     * return the string encoding for the selected time range (with additional TR info)
     * @param value
     */
    AnalyticTimeRangeFE.prototype.encodeToString = function () {
        var timeRangeInfo = this.timeRangeInfo,
            ordinal = this.timeRangeOrdinal;
        var enumVal = baja.$("analytics:UxAnalyticTimeRangeType").make(ordinal);
        var trInfo = timeRangeInfo[enumVal.getTag()];
        if (ordinal === 0) {
            var ret = "from" + trInfo.fromAlignment;
            if (trInfo.fromOffsetEnabled) {
                ret += trInfo.fromOffset;
            }
            ret += "To";
            ret += trInfo.toOffset;
            return ret;
        } else if (ordinal === 1 || ordinal === 2) {
            return enumVal.getTag() + trInfo.val + trInfo.duration;
        } else if (ordinal === 3) {
            return encodeURIComponent(trInfo.startTime.encodeToString() + ";" + trInfo.endTime.encodeToString());
        }
        return enumVal.getTag(); // may be a safe default
    };
    /**
     * Decode the time range info from the string
     * @param value
     */
    AnalyticTimeRangeFE.prototype.decodeFromString = function (value) {
        switch (value.charAt(0)) {
            case 't': //this,today
            case 'y': //yearToDate
            case 'm': //monthToDate
            case 'w': //weekToDate
            case 'l': //last
            case 'a':
                //all
                this.parseConstants(value);
                break;
            case 'c': //current
            case 'p': // Previous
            case 'n':
                // Next
                this.parsePCNParts(value);
                break;
            case 'f':
                //from
                this.parseFromParts(value);
                break;
        }
        if (value.indexOf(';') > 0) // This is a date range (from and to (exclusive))
            {
                this.timeRangeOrdinal = 3;
                var startend = value.split(";");
                var tr = this.timeRangeInfo.timeRainge = {};
                tr.startTime = analyticsUtil.getAbsTime(moment(startend[0]));
                tr.endTime = analyticsUtil.getAbsTime(moment(startend[1]));
            }
    };
    /**
     * Parse current, previous and next
     * @param value
     * @returns {*}
     */
    AnalyticTimeRangeFE.prototype.parsePCNParts = function (s) {
        var ret = new Array(3);
        if (s.indexOf("previous") === 0) {
            ret[0] = "previous";
            this.timeRangeOrdinal = 2;
        } else if (s.indexOf("current") === 0) {
            ret[0] = "current";
            this.timeRangeOrdinal = 1;
        } else if (s.indexOf("next") === 0) {
            ret[0] = "next";
        } else {
            return Error("Invalid option");
        }
        var i = ret[0].length;
        var j = i;
        var len = s.length;
        while (j < len) {
            if ($.isNumeric(s.charAt(j))) {
                j++;
            } else {
                break;
            }
        }
        ret[1] = s.substring(i, j);
        ret[2] = s.substring(j);
        this.timeRangeInfo[ret[0]] = {
            ordinal: this.timeRangeOrdinal,
            val: parseInt(ret[1]),
            duration: ret[2]
        };
    };
    /**
     * Parse the from parts
     * @param s
     * @returns {*}
     */
    AnalyticTimeRangeFE.prototype.parseFromParts = function (str) {
        this.timeRangeOrdinal = 0;
        var trInfo = this.timeRangeInfo.from = {},
            list = [];
        if (str.indexOf("from") === 0) {
            list.push("from");
        } else {
            throw new Error("Invalid time : " + str);
        }
        var to = str.indexOf("To");
        if (to < 0) {
            throw new Error("Invalid time : " + str);
        }
        var i = str.indexOf("P", 6);
        if (i > 0 && i < to) {
            var ch = str.charAt(i - 1);
            if (ch === '+' || ch === '-') {
                i--;
            }
            trInfo.fromAlignment = str.substring(4, i); //alignment
            trInfo.fromOffset = str.substring(i, to); //start offset
        } else {
            trInfo.fromAlignment = str.substring(4, to); //alignment
        }
        if (str.indexOf("Now", to) < 0) {
            trInfo.toOffset = str.substring(to + 2);
        } else {
            trInfo.toOffset = "Now";
        }
    };
    /**
     * These time range strings do not need any further processing
     * @param value
     */
    AnalyticTimeRangeFE.prototype.parseConstants = function (str) {
        switch (str.charAt(0)) {
            case 'y':
                //yearToDate
                if (str.equals("yesterday")) {
                    this.timeRangeOrdinal = 8;
                }
                if (str.equals("yearToDate")) {
                    this.timeRangeOrdinal = 6;
                }
                break;
            case 'm':
                //monthToDate
                if (str.equals("monthToDate")) {
                    this.timeRangeOrdinal = 5;
                }
                break;
            case 'w':
                //weekToDate
                if (str.equals("weekToDate")) {
                    this.timeRangeOrdinal = 4;
                }
                break;
            case 't':
                //this
                if (str.equals("today")) {
                    this.timeRangeOrdinal = 7;
                }
                if (str.equals("thisWeek")) {
                    this.timeRangeOrdinal = 9;
                }
                if (str.equals("thisMonth")) {
                    this.timeRangeOrdinal = 11;
                }
                if (str.equals("thisYear")) {
                    this.timeRangeOrdinal = 13;
                }
                break;
            case 'l':
                //last
                if (str.equals("lastWeek")) {
                    this.timeRangeOrdinal = 10;
                }
                if (str.equals("lastMonth")) {
                    this.timeRangeOrdinal = 12;
                }
                if (str.equals("lastYear")) {
                    this.timeRangeOrdinal = 14;
                }
                break;
            case 'a':
                if (str.equals("all")) {
                    this.timeRangeOrdinal = 15;
                }
                break;
        }
    };

    return AnalyticTimeRangeFE;
});
