/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/AnalyticDataEditorFE
 */
define(['baja!', 'jquery', 'Promise', 'bajaux/Widget', 'dialogs', 'bajaux/events', 'hbs!nmodule/analytics/rc/chart/templates/fe/analyticDataEditor', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'lex!analytics'], function (baja, $, Promise, Widget, dialogs, events, analyticDataEditor, BaseEditor, lexicon) {

  'use strict';

  var lex = lexicon[0];

  /**
   * A field editor for selecting tags from the station.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:analytics/rc/chart/fe/AnalyticDataEditorFE
   */
  var AnalyticDataEditorFE = function AnalyticDataEditorFE() {
    BaseEditor.apply(this, arguments);
  };
  // Inheriting AnalyticDataEditorFE from BaseEditor
  AnalyticDataEditorFE.prototype = Object.create(BaseEditor.prototype);
  // Setting the constructor
  AnalyticDataEditorFE.prototype.constructor = AnalyticDataEditorFE;

  /**
   * Returns the Data element (Tag string)
   * e.g n:history , hs:ahu
   * @private
   * @returns {jQuery}
   */
  AnalyticDataEditorFE.prototype.$getDataElement = function () {
    return this.jq().children('input');
  };

  /**
   * Creates a text box with a Data selection picker.
   * The data selection picker has a list box that shows all the available
   * tags for the selected node or all tags in the station.
   *
   * It uses NiagaraRpc to retrieve data from station.
   * @param {jQuery} dom
   */
  AnalyticDataEditorFE.prototype.doInitialize = function (dom) {
    var that = this;
    dom.append('<input type="text" class="dataSelectValue">&nbsp;<button class="dataSelectLauncher" style="height: 20px;">' + lex.getSafe("dataselector.command") + '</button>');
    $("input.dataSelectValue").keyup(function () {
      that.jq().trigger(events.MODIFY_EVENT);
    });
    dom.on('click', 'button', function () {
      dialogs.showOkCancel({
        title: lex.get("title.dataSelector"), content: function content(dlg, jq) {
          baja.rpc({
            ord: "service:analytics:AnalyticService",
            args: [{ 'selNode': baja.Ord.DEFAULT }],
            methodName: "getTagDataForBox"
          }).then(function (data) {
            jq.html(analyticDataEditor({
              tags: data
            }));
            that.registerEvents(jq, dom);
          }).catch(function (err) {
            jq.html(analyticDataEditor({
              tags: {}
            }));
          }).finally(function () {
            // set defaults if applicable
            $("select.dataList", jq).val($("input.selectedValue").val());
          });
        }
      }).ok(function (obj) {
        $("input.dataSelectValue", dom).val(obj.jq().find("input.selectedValue").val());
        // Explicitly invoke in the best benifit of FE consumers.
        that.jq().trigger(events.MODIFY_EVENT);
      });
    });
    return new Promise(function (resolve, reject) {
      resolve(that.jq());
    });
  };
  /**
   * Register the events for selection of data tag
   */
  AnalyticDataEditorFE.prototype.registerEvents = function (jq, dom) {
    var inputElement = jq.find("input.filterText");
    var optionJq = $("select.dataList > option", jq);
    inputElement.keyup(function () {
      var valThis = $(this).val().toLowerCase();
      if (valThis === "") {
        optionJq.show();
      } else {
        optionJq.each(function () {
          var optionElement = $(this);
          var text = optionElement.text().toLowerCase();
          if (text.indexOf(valThis) >= 0) {
            optionElement.appendTo($("select.dataList", jq));
          } else {
            optionElement.remove();
          }
        });
      }
    });

    var selectBox = jq.find("select.dataList");
    var selectedValue = jq.find("input.selectedValue");
    selectBox.change(function () {
      var optionSelected = $("option:selected", this);
      var value = optionSelected.val();
      selectedValue.val(value);
    });

    $(".clearFilterText", jq).click(function () {
      $(".filterText", jq).val("");
      $('select.dataList > option', jq).show();
    });
  };
  /**
   * Loads a `baja.String` data value into the input element. I
   * @param {baja.String} value
   * @returns {Promise} promise to be resolved if the value is loaded,
   * or rejected if the `Data Value` could not be encoded to string
   */
  AnalyticDataEditorFE.prototype.doLoad = function (value) {
    this.$getDataElement().val(value);
  };

  /**
   * Reads the data value from the date input element and constructs a
   * `Data` from it.
   * @returns {String} the read `Data` value
   */
  AnalyticDataEditorFE.prototype.doRead = function () {
    var val = this.$getDataElement().val();
    return val;
  };

  /**
   * Enables or disables the data input element.
   * @param {Boolean} enabled
   */
  AnalyticDataEditorFE.prototype.doEnabled = function (enabled) {
    this.$getDataElement().prop('disabled', !enabled || this.isReadonly());
  };

  /**
   * Enables or disables the data input element.
   * @param {Boolean} readonly
   */
  AnalyticDataEditorFE.prototype.doReadonly = function (readonly) {
    this.$getDataElement().prop('disabled', readonly || !this.isEnabled());
  };

  return AnalyticDataEditorFE;
});
