/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * SpectrumChart.js
 * This javascript creates a SpectrumChart/HeatMap using d3 javascript.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'underscore', 'd3', 'moment', 'nmodule/analytics/rc/chart/base/AnalyticD3BaseChart', 'nmodule/analytics/rc/chart/types/AnalyticD3LineChart', 'nmodule/analytics/rc/chart/types/AnalyticD3StackedBarChart', 'nmodule/analytics/rc/chart/equipop/AnalyticsTabConfigEqOpModel', 'nmodule/analytics/rc/chart/fe/EqOpConfigurationSettingsFE', 'nmodule/webEditors/rc/wb/util/ICollectionSupport', 'lex!analytics', 'baja!analytics:AnalyticsTabConfigEqOpModel', 'css!nmodule/analytics/rc/chart/equipop/equipOperationChart'], function (baja, Widget, subscriberMixIn, Promise, $, _, d3, moment, AnalyticsD3BaseChart, AnalyticD3LineChart, AnalyticD3StackedBarChart, AnalyticsTabConfigEqOpModel, EqOpConfigurationSettingsFE, ICollectionSupport, lexicon) {

  "use strict";

  var lex = lexicon[0];
  var END_OF_TIME = baja.$("baja:AbsTime").make({
    date: baja.$("baja:Date").make({ year: 9999, month: 11, day: 31 }),
    time: baja.$("baja:Time").make({ hour: 23, min: 59 }),
    offset: 0
  });

  /**
   * @class
   * @alias module:nmodule/analytics/rc/chart/equipop/EquipmentOperationChart
   * @extends module:nmodule/analytics/rc/chart/base/AnalyticsD3BaseChart
   * @mixes module:nmodule/webEditors/rc/wb/util/ICollectionSupport
   * @param {Object} [params]
   */
  var EquipmentOperationChart = function EquipmentOperationChart(params) {
    AnalyticsD3BaseChart.call(this, $.extend({}, params));
    ICollectionSupport(this);

    this.minTime = END_OF_TIME.getJsDate();
    this.maxTime = baja.AbsTime.DEFAULT.getJsDate();
  };

  EquipmentOperationChart.prototype = Object.create(AnalyticsD3BaseChart.prototype);
  EquipmentOperationChart.prototype.constructor = EquipmentOperationChart;

  EquipmentOperationChart.prototype.processData = function (chartSettingsCollection) {
    var that = this,
        ser;
    that.series = [];
    chartSettingsCollection.map(function (model, index) {
      if (index === 0) {
        that.renderInterval = model.getInterval();
      }
      ser = model.getAnalyticTrendArray();
      var processedSeries = {
        color: model.getBrush(),
        name: model.getSeriesName(),
        unitSymbol: model.getUnit().getSymbol(),
        ord: model.getOrd()
      };

      processedSeries.data = ser.map(function (e) {
        var time = e.date;
        if (time.getMillis() > that.maxTime.getTime()) {
          that.maxTime = time.getJsDate();
        }
        if (time.getMillis() < that.minTime.getTime()) {
          that.minTime = time.getJsDate();
        }
        var date = e.date.getJsDate();
        return {
          x: date,
          y: e.value > 0 ? 1 : 0,
          metaData: { interpolationStatus: e.interpolationStatus },
          trendFlags: e.trendFlags
        };
      });
      that.series.push(processedSeries);
    });
  };

  EquipmentOperationChart.prototype.needsBreakOnOneIteration = function () {
    return true;
  };
  /**
   *
   * @returns {[string,string]}
   */
  EquipmentOperationChart.prototype.getColumnNames = function () {
    var arr = ['Date Time'],
        that = this,
        chartModelList = that.chartModelList;
    _.each(chartModelList, function (chartModel, index) {
      var sn = baja.SlotPath.unescape(chartModel.getSeriesName());
      arr.push(sn);
      arr.push(sn + " " + lex.get("chart.interpolationStatus.label"));
    });
    that.columnNames = arr;
    return arr;
  };

  /**
   *
   * @returns {[string,string]}
   */
  EquipmentOperationChart.prototype.getColumnKeys = function () {
    var arr = ['date'],
        that = this,
        chartModelList = that.chartModelList;
    _.each(chartModelList, function (chartModel, index) {
      var sn = chartModel.getSeriesName();
      arr.push(sn);
      arr.push(sn + "interpolationStatus");
    });
    that.columnKeys = arr;
    return arr;
  };

  EquipmentOperationChart.prototype.getXAxisLabel = function (d) {
    return lex.get("chart.equipmentop.xaxis.label");
  };

  EquipmentOperationChart.prototype.getYAxisLabel = function (d) {
    return lex.get("chart.equipmentop.yaxis.label");
  };

  EquipmentOperationChart.prototype.getXAxis = function () {
    var that = this;
    that.xScale = d3.time.scale().range([0, that.availableWidth()]);

    var minMaxDuration = that.getXMinMax();
    that.minDuration = minMaxDuration.min;
    that.maxDuration = minMaxDuration.max;

    if (that.maxDuration !== 0) {
      that.xScale.domain([that.minDuration, that.series[0].data[1] && that.series[0].data[1].x ? moment(that.maxDuration.getTime() + (that.series[0].data[1].x.getTime() - that.series[0].data[0].x.getTime())).toDate() : moment(that.maxDuration.getTime()).add(that.getRenderInterval(), "ms").toDate()]);
    } else {
      that.xScale.domain([0, 1440 * 60 * 1000]);
    }
    return that.buildXAxis(that.xScale);
  };

  /**
   * Returns an empty Y Axis and the D3StackedBar Chart handles the YAxis manually.
   */
  EquipmentOperationChart.prototype.getYAxis = function () {
    return d3.svg.axis().tickValues([]);
  };

  /**
   *
   * @param that
   * @param inDat
   * @param interpolationStatus
   * @return {*}
   */
  function getInterpolTooltipTxt(that, inDat, interpolationStatus, seriesName) {
    var ttipString;
    if (that.showInterpolationStatus) {
      ttipString = lex.getSafe({
        key: "chart.equipmentop.tooltip.y",
        args: [baja.SlotPath.unescape(seriesName), inDat === undefined ? lex.getSafe('baseChart.nodatafound.message') : inDat.y === 1 ? lex.getSafe('chart.equipmentop.tooltip.value.on') : lex.getSafe('chart.equipmentop.tooltip.value.off'), lex.getSafe("chart.equipmentop.tooltip.interpolation") + interpolationStatus]
      });
    } else {
      ttipString = lex.getSafe({
        key: "chart.equipmentop.withoutInterPolTooltip.y",
        args: [baja.SlotPath.unescape(seriesName), inDat === undefined ? lex.getSafe('baseChart.nodatafound.message') : inDat.y === 1 ? lex.getSafe('chart.equipmentop.tooltip.value.on') : lex.getSafe('chart.equipmentop.tooltip.value.off')]
      });
    }
    return ttipString;
  }

  EquipmentOperationChart.prototype.getToolTipHtml = function (d) {
    var that = this;
    if (d === undefined) {
      return;
    }

    var xValue = d.x,
        interpolationStatus = d.metaData.interpolationStatus;
    var yList = "";
    yList += lex.getSafe({
      key: "chart.equipmentop.tooltip.time",
      args: [xValue.toLocaleString()]
    });
    yList += lex.getSafe("chart.equipmentop.tooltip.value");

    that.series.map(function (d) {
      var seriesName = d.name || "";
      var marker = "<div class='tooltip-marker'" + " style='background: " + d.color + "'></div>";
      var inDat = _.find(d.data, function (dat) {
        return dat.x.getTime() === xValue.getTime();
      });
      yList += "<div>" + marker + getInterpolTooltipTxt(that, inDat, interpolationStatus, seriesName) + "</div>";
    });
    return yList;
  };

  /**
   * Get the chart margins
   * @type {{top: number, right: number, bottom: number, left: number}}
   */
  EquipmentOperationChart.prototype.chartMargins = function () {
    return {
      top: 20,
      right: 50,
      bottom: 60,
      left: 70
    };
  };

  EquipmentOperationChart.prototype.supportedCharts = function () {
    return [AnalyticD3StackedBarChart];
  };

  /**
   * This method should be overridden by subclasses to return respective chart types.
   * The default implementation retruns "none".
   * @returns {string}
   */
  EquipmentOperationChart.prototype.getSupportedExportTypes = function () {
    return ["eqopchart", "csv"];
  };

  /**
   * Base model for configuration.
   * Children should override this method and define their tab model. Then only their configuration will be seen.
   * @returns {*}
   */
  EquipmentOperationChart.prototype.getTabModelType = function () {
    return AnalyticsTabConfigEqOpModel;
  };

  /**
   * Base model java class for configuration.
   * Child classes should override this method and mention their configuration class.
   * @returns {*}
   */
  EquipmentOperationChart.prototype.getConfigTabParamType = function () {
    return baja.$("analytics:AnalyticsTabConfigEqOpModel");
  };

  /**
   * Get Settings Editor Type
   * @returns {*}
   */
  EquipmentOperationChart.prototype.getConfigurationTabType = function () {
    return EqOpConfigurationSettingsFE;
  };

  /**
   * Abstract method allows the child class to implement the logic to update their tab model based on the web object.
   * @param chartObj
   */
  EquipmentOperationChart.prototype.updateChartConfiguration = function (data) {
    var that = this;
    that.getTabConfigDataModel().setOffStatus(data.getOffStatus());
    // that.render(that.chartModelList);
  };

  EquipmentOperationChart.prototype.getName = function () {
    return "EquipmentOperationChart";
  };

  EquipmentOperationChart.prototype.doDestroy = function () {
    this.jq().find('.chartArea').removeData();
    this.jq().find('.chartContainer').removeData();
    return Promise.all([d3.select('.chartArea').selectAll('svg').remove(), d3.select('.chartContainer').selectAll('svg').remove()]);
  };

  return EquipmentOperationChart;
});
