/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
* @module nmodule/analytics/rc/chart/base/AnalyticsBaseChart
*/
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'moment', 'dialogs', 'underscore', 'd3', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'hbs!nmodule/analytics/rc/chart/templates/base/analyticsBaseChartTmpl', 'hbs!nmodule/analytics/rc/chart/templates/fe/analyticBaseChartMasterTabTemplate', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/chart/base/AnalyticsBaseWidget', 'nmodule/analytics/rc/chart/utils/AnalyticsChartDragDropUtils', 'nmodule/analytics/rc/chart/utils/AnalyticsDataUtils', 'nmodule/analytics/rc/chart/fe/AnalyticUxWebChartSettingsFE', 'nmodule/analytics/rc/chart/fe/AnalyticUxConfigurationSettingsFE', 'nmodule/analytics/rc/chart/base/AnalyticsCommandBar', 'nmodule/analytics/rc/chart/base/AnalyticsBaseDataModel', 'bajaux/events', 'nmodule/analytics/rc/chart/base/analyticEvents', 'baja!analytics:AnalyticUxWebChartParams,' + 'analytics:UxAnalyticTimeRangeType,' + 'analytics:Interval,' + 'analytics:Combination,' + 'analytics:AnalyticsTabConfigBaseModel', 'lex!analytics', 'css!nmodule/analytics/rc/chart/styles/ChartStyles', 'css!nmodule/analytics/rc/chart/styles/TabStyles', 'css!nmodule/js/rc/dialogs/dialogs'], function (baja, Widget, subscriberMixIn, Promise, $, moment, dialogs, _, d3, fe, feDialogs, baseTemplate, analyticBaseChartMasterTabTemplate, analyticsUtil, AnalyticsBaseWidget, AnalyticsChartDragDropUtils, AnalyticsDataUtils, AnalyticUxWebChartSettingsFE, AnalyticUxConfigurationSettingsFE, AnalyticsCommandBar, AnalyticsBaseDataModel, events, analyticEvents, types, lexicon) {

  "use strict";

  var lex = lexicon[0];
  /**
   * Base view for all web analytics charts
   * 
   * @class
   * @alias module:nmodule/analytics/rc/chart/base/AnalyticsBaseChart
   * @extends module:nmodule/analytics/rc/chart/base/AnalyticsBaseWidget
   */
  var AnalyticsBaseChart = function AnalyticsBaseChart(params) {
    AnalyticsBaseWidget.call(this, $.extend({}, params));
    subscriberMixIn(this);
    this.chartModelList = [];
    // Requests are pooled by their unique key and checked for their reoccurance.
    this.uniqueKeyList = [];
    this.series = [];
    this.$isFile = false;
    this.$fileName = "";
    this.$widgetCommandBarRendered = false;
    this.$isChartBeingDrawn = false;
    this.$isChartDrawn = false;
    this.$svg = undefined;
    this.$selectedChartType = undefined;
    this.$tabConfigDataModel = undefined;
    this.isInitCommandBar = true;
    this.columnNames = ['Date', 'Value'];
    this.columnKeys = ['x', 'y'];
  };
  // Inheriting Analytics Base Chart from Widget
  AnalyticsBaseChart.prototype = Object.create(AnalyticsBaseWidget.prototype);
  // Setting the constructor
  AnalyticsBaseChart.prototype.constructor = AnalyticsBaseChart;

  /**
   * This method does the necessary initialization for the charts. This need not
   * be overridden for any of the analytic charts unless a custom behavior is needed.
   *
   * The list of activities performed  are :- <br>
   *   <ul>
   *       <li> Initialize a drag drop context on the chart area </li>
   *       <li> Provides a default drop handler <li>
   *       <li> Invoke the initialization callback for sub classes </li>
   *       <li> Load the base shell that would be common across all the charts </li>
   *       <li> handle the events for the settings </li>
   *   </ul>
   * Adds a `drag drop ` listener to this view's root element.
   */
  AnalyticsBaseChart.prototype.doInitialize = function (element) {
    var that = this;
    return AnalyticsBaseWidget.prototype.doInitialize.apply(that, [element]).then(function (load) {
      element.html(baseTemplate());
      registerWidgetChangeEvent(that);
      var ordPromise = baja.Ord.make('station:|slot:/Services/AnalyticService/chartRenderCapacity').get().then(function (chartRenderCapacity) {
        that.chartRenderCapacity = chartRenderCapacity;
        return chartRenderCapacity;
      }).catch(function () {
        return Promise.resolve();
      });
      that.windowSize = {
        width: window.innerWidth,
        height: window.innerHeight
      };
      registerResizeEvent(that);
      if (load && typeof load === "function") {
        return Promise.all([load(), ordPromise]);
      }
      return Promise.all([ordPromise]);
    });

    // Now render the base template
  };

  /**
   *  <pre> doLoad </pre> method on this base chart view loads the chart view.
   *  The list of steps include :-
   *      <ul>
   *          <li> Read the data from various possible sources (File, Component or PX View)
   *          <br>
   *              In all of the above cases the mapping is :-
   *              <ul>
   *                  <li> Export File corresponds to box:BoxFile </li>
   *                  <li> Px View settings correspond to box:BoxTable</li>
   *                  <li> Component drag drop corresponds to Object with ord </li>
   *              </ul>
   *          </li>
   *      </ul>
   */
  AnalyticsBaseChart.prototype.doLoad = function (source) {
    var that = this;
    return AnalyticsBaseWidget.prototype.doLoad.apply(this, [source]).then(function (chartSource) {
      if (!that.$isDragDropInvoked && (that.$isLoaded || _.isEmpty(chartSource))) {
        return Promise.resolve(true);
      }
      that.$isDragDropInvoked = false;
      if (that.source === "report") {
        return that.render(that.chartModelList);
      } else {
        var sPromise = AnalyticsDataUtils.getModel(that, chartSource, that.getDefaultSettings());
        sPromise.then(function (inModel) {
          var settingsModel = inModel[0];
          // Make a local collection of unique keys. Kick the doLoad call, if the unique key already exists.
          if (_.contains(that.uniqueKeyList, settingsModel[0].getUniqueKey())) {
            return Promise.resolve();
          } else {
            if (settingsModel[0].getUniqueKey()) {
              that.uniqueKeyList.push(settingsModel[0].getUniqueKey());
            }
          }

          if (!that.isFile() && AnalyticsDataUtils.isFile(settingsModel[0])) {
            that.$isFile = true;
            that.$fileName = settingsModel[0].getDataSourceName();
          }

          if (_.isArray(settingsModel)) {
            that.chartModelList.push.apply(that.chartModelList, settingsModel);
          }

          if (!_.isEmpty(inModel[1])) {
            that.$tabConfigDataModel.$rebuild(inModel[1]);
          }

          // Get the chart data and trigger a chart render
          return that.$doLoad();
        });
        return sPromise;
      }
    });
  };

  /**
   * Will finish the tasks before doLoad finishes.
   */
  AnalyticsBaseChart.prototype.finishTasks = function () {
    var that = this;
    //Set the text size for all text elements in svg.
    that.svg().selectAll("text").style("font-size", that.getTabConfigDataModel().getFontSize() + "px");
    //Set the text size for time range element text.
    that.jq().find(".timeRangeElem, .trCmdBar, .ux-chart-legend-text").attr("style", "font-size:" + that.getTabConfigDataModel().getFontSize() + "px");
    // If the available height is greater than zero then set the icon sizes.
    if (this.availableHeight() > 0) {
      this.legendIconSize = this.legendIconSize === 0 ? this.getLegendIconScale()(this.availableHeight()).toFixed(0) : this.legendIconSize;
      this.generalIconSize = this.generalIconSize === 0 ? this.getGeneralIconScale()(this.availableHeight()).toFixed(0) : this.generalIconSize;
    }
    if (that.showLegend) {
      that.updateLegend(that.chartModelList);
    }
  };

  /**
   * Load the data form the collection
   * @param chartSettingsCollection Array
   */
  AnalyticsBaseChart.prototype.render = function (chartSettingsCollection) {
    var that = this;
    var dataPresent = false;
    that.chartModelList.map(function (model) {
      if (model) {
        if (model.getAnalyticTrendArray().length > 0) {
          dataPresent = true;
        }
      }
    });

    that.$isChartDrawn = false;
    that.$isChartBeingDrawn = true;
    if (that.needsCleanSlate()) {
      $(".chartArea", that.jq()).unbind().empty();
    }

    that.initSvg();
    var drawChartPromise = that.reDraw(chartSettingsCollection);
    return Promise.resolve(drawChartPromise).then(function () {
      that.$isChartDrawn = true;
      that.$isChartBeingDrawn = false;
      that.finishTasks();
      // triggerChartChangedEvent(that);
    });
  };
  /**
   * Empty the data area and put some text
   */
  AnalyticsBaseChart.prototype.renderEmpty = function () {
    var chartArea = $(".chartArea", this.jq());
    var message = $("<div class='no-data-text'>"),
        text = lex.getSafe("analytics.chart.nodata");
    message.html(text);
    chartArea.unbind().empty().append(message);
    return text;
  };
  /**
   * Update the chart info
   * @param chartSource
   */
  AnalyticsBaseChart.prototype.updateLegend = function (chartSettingsCollection) {
    var that = this,
        interimObj = {},
        legendArea = this.jq().find("div.chart-metadata-area"),
        chartArea = $(".chartArea", this.jq());
    // Empty the existing legend area
    legendArea.empty();
    legendArea.css('width', '20%');
    chartArea.css('width', '78%');
    chartSettingsCollection.map(function (e) {
      interimObj[analyticsUtil.getBaseOrd(e.getOrd())] = e;
    });

    chartSettingsCollection.map(function (model) {
      var utx = model.getUnitDisplay();
      var unitText = _.escape(utx ? " (" + utx + ")" : utx),
          iconSize = _.escape(that.getLegendIconSize() + "px"),
          textSize = _.escape(that.getTabConfigDataModel().getFontSize() + "px"),
          titleText = _.escape(analyticsUtil.formatLegendTooltip(model));
      var wrapper = $("<div>");
      wrapper.css('padding', '2px');
      wrapper.append("<div class='ux-chart-legend-color' title='" + titleText + "' style='background-color:" + model.getBrush() + ";  width: " + iconSize + "; height: " + iconSize + "'></div>");
      wrapper.append("<div class='ux-chart-legend-text' title='" + titleText + "' style='font-size: " + textSize + "'>" + baja.SlotPath.unescape(model.getSeriesName()) + (_.isEmpty(model.getAnalyticTrendArray()) ? " (" + lex.getSafe("baseChart.nodatafound.message") + ")" : "") + unitText + "</div>");
      wrapper.append("<div style='clear:both'></div>");
      wrapper.appendTo(legendArea);
    });
  };

  /**
   * Get Settings Editor Type
   * @returns {*}
   */
  AnalyticsBaseChart.prototype.getSettingsEditorType = function () {
    return AnalyticUxWebChartSettingsFE;
  };

  /**
   * Get Configuration Editor Type.
   * Child charts should implement their own configuration type.
   * @returns {*}
   */
  AnalyticsBaseChart.prototype.getConfigurationTabType = function () {
    return AnalyticUxConfigurationSettingsFE;
  };

  /**
   *
   * @returns {undefined|*}
   */
  AnalyticsBaseChart.prototype.initSvg = function () {
    throw new Error("Not Implemented");
  };

  /**
   *
   */
  AnalyticsBaseChart.prototype.getToolTipDiv = function () {
    throw new Error("Not Implemented");
  };

  AnalyticsBaseChart.prototype.svg = function () {
    throw new Error("Not Implemented");
  };

  AnalyticsBaseChart.prototype.availableWidth = function () {
    return parseInt(this.getChartPaneWidth()) - this.chartMargins().left - this.chartMargins().right;
  };

  AnalyticsBaseChart.prototype.availableHeight = function () {
    return parseInt(this.getChartPaneHeight()) - this.chartMargins().top - this.chartMargins().bottom;
  };

  AnalyticsBaseChart.prototype.initXAxis = function (text) {
    throw new Error("Not Implemented");
  };

  AnalyticsBaseChart.prototype.initYAxis = function (text) {
    throw new Error("Not Implemented");
  };

  AnalyticsBaseChart.prototype.getXAxis = function () {
    throw new Error("Unsupported Operation");
  };

  AnalyticsBaseChart.prototype.getYAxis = function () {
    throw new Error("Unsupported Operation");
  };

  AnalyticsBaseChart.prototype.supportedCharts = function () {
    throw new Error("Unsupported Operation");
  };

  AnalyticsBaseChart.prototype.getSelectedChartType = function () {
    return this.$selectedChartType || this.supportedCharts()[0];
  };

  AnalyticsBaseChart.prototype.setSelectedChartType = function (selectedChartType) {
    this.$selectedChartType = selectedChartType;
  };

  AnalyticsBaseChart.prototype.getSeriesLabel = function () {};

  AnalyticsBaseChart.prototype.getXMinMax = function () {
    var that = this;
    return analyticsUtil.getXMinMax(that.series);
  };
  /**
   * Get the formatted tooltip value. Needs to be implemented
   * @param value
   */
  AnalyticsBaseChart.prototype.getFormattedTooltipValue = function (value) {};

  AnalyticsBaseChart.prototype.needsBreakOnOneIteration = function () {
    return false;
  };

  /**
   * Register a control widget changed event.
   * @param that
   */
  function registerWidgetChangeEvent(that) {
    that.jq().off(analyticEvents.CONTROL_WIDGET_CHANGED);
    that.jq().on(analyticEvents.CONTROL_WIDGET_CHANGED, function (event, chartSource) {
      that.load(chartSource);
    });
  }

  /**
   * Handler for resize event.
   * @param event */
  function resizeEventHandler(event) {
    var that = event.data.that;
    if (that.windowSize.width !== window.innerWidth && that.windowSize.height !== window.innerHeight) {
      var milliseconds = event.data.milliseconds;
      // If the chart is being redrawn or the chart is not drawn, do not let the redraw operation.
      if (!that.$isChartBeingDrawn && that.$isChartDrawn) {
        setTimeout(function () {
          that.windowSize = {
            width: window.innerWidth,
            height: window.innerHeight
          };
          that.$isChartBeingDrawn = true;
          that.$isChartDrawn = false;
          if (_.every(that.chartModelList, function (m) {
            return _.isEmpty(m.getAnalyticTrendArray());
          })) {
            var message = that.renderEmpty();
            return Promise.reject(message);
          }
          if (that.needsCleanSlate()) {
            $(".chartArea", that.jq()).unbind().empty();
          }
          that.initSvg();
          var reDrawRequest = true,
              redrawChartPromise = that.draw(that.chartModelList, reDrawRequest);
          Promise.resolve(redrawChartPromise).then(function () {
            that.$isChartBeingDrawn = false;
            that.$isChartDrawn = true;
            that.finishTasks();
            // triggerChartChangedEvent(that);
          });
        }, milliseconds);
      }
    } else {}
  }

  /**
   * Register resize to get called upon resize and chart expanded events.
   * @param that
   */
  function registerResizeEvent(that) {
    $(window).off("resize");
    $(window).resize({ milliseconds: 0, that: that }, resizeEventHandler);
    that.jq().off(analyticEvents.CHART_EXPANDED);
    that.jq().on(analyticEvents.CHART_EXPANDED, {
      milliseconds: 0,
      that: that
    }, resizeEventHandler);
  }

  /**
   *
   * @returns {{columnNames: [string,string], columnKey: [string,string]}}
   */
  AnalyticsBaseChart.prototype.getSeriesAnalogy = function () {
    var that = this;
    return {
      columnNames: that.getColumnNames(),
      columnKeys: that.getColumnKeys()
    };
  };

  AnalyticsBaseChart.prototype.setColumnNames = function (colNames) {
    var that = this;
    that.columnNames = colNames;
  };

  AnalyticsBaseChart.prototype.setColumnKeys = function (colKeys) {
    var that = this;
    that.columnKeys = colKeys;
  };

  AnalyticsBaseChart.prototype.getColumnNames = function () {
    var that = this;
    return that.columnNames;
  };

  AnalyticsBaseChart.prototype.getColumnKeys = function () {
    var that = this;
    return that.columnKeys;
  };

  AnalyticsBaseChart.prototype.getTableDataStructure = function () {
    return 'data';
  };

  AnalyticsBaseChart.prototype.isClrToBeDisplayedInRpt = function () {
    return false;
  };

  AnalyticsBaseChart.prototype.getColorCodedColKeys = function () {
    return [];
  };

  return AnalyticsBaseChart;
});
