/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * SpectrumChart.js
 * This javascript creates a SpectrumChart/HeatMap using d3 javascript.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'd3', 'underscore', 'nmodule/analytics/rc/chart/base/AnalyticC3BaseChart', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/webEditors/rc/wb/util/ICollectionSupport', 'lex!analytics', 'css!nmodule/analytics/rc/chart/averageprofile/averageProfileChart'], function (baja, Widget, subscriberMixIn, Promise, $, d3, _, AnalyticC3BaseChart, analyticsUtil, ICollectionSupport, lexicon) {

  "use strict";

  var lex = lexicon[0];

  /**
   *
   * @class
   * @alias module:nmodule/analytics/rc/chart/averageprofile/AverageProfileChart
   * @extends module:nmodule/analytics/rc/chart/base/AnalyticC3BaseChart
   * @mixes module:nmodule/webEditors/rc/wb/util/ICollectionSupport
   * @param {Object} [params]
   */
  var AverageProfileChart = function AverageProfileChart(params) {
    AnalyticC3BaseChart.call(this, $.extend({}, params));
    ICollectionSupport(this);
    this.processedSeriesMap = {};
    this.interval = -1;
    this.maxTimeOfDay = -1; //safe bet
    this.intervalsPerDay = 0;
    this.timeofdayArray = [];
    this.minValue = baja.Float.MAX_VALUE;
    this.maxValue = baja.Float.MIN_VALUE;
  };

  AverageProfileChart.prototype = Object.create(AnalyticC3BaseChart.prototype);
  AverageProfileChart.prototype.constructor = AverageProfileChart;

  /**
   * Render the pie
   * @param chartSettingsCollection
   */

  AverageProfileChart.prototype.processData = function (chartSettingsCollection, isRedrawRequest) {
    var that = this;
    return new Promise(function (resolve, reject) {
      that.series = [];
      _.each(chartSettingsCollection, function (model, i) {
        var series = model.getAnalyticTrendArray();
        if (!isRedrawRequest) {
          series = analyticsUtil.normalize(model, series);
        }
        var processedSeries = that.processSeries(series);
        that.processedSeriesMap[model.getSeriesName() || "Series " + i] = {
          series: processedSeries,
          brush: model.getBrush(),
          units: model.getUnit().getSymbol()
        };
        that.series.push({
          name: model.getSeriesName(),
          data: processedSeries,
          color: model.getBrush(),
          unitSymbol: model.getUnitDisplay(),
          ord: model.getOrd(),
          unit: model.getUnitDisplay(),
          chartType: model.getChartType(),
          showOnY2: model.getShowOnY2()
        });
      });
      resolve(analyticsUtil.getRenderableC3Data(that, that.series));
    });
  };

  /**
   *
   * @param data
   * @returns {number}
   */
  AverageProfileChart.prototype.getXData = function (data) {
    return data.x;
  };

  /**
   *
   * @param data
   * @returns {number}
   */
  AverageProfileChart.prototype.getYData = function (data) {
    return data.y;
  };

  /**
   *
   * @param x
   * @returns {*}
   */
  AverageProfileChart.prototype.getXTickFormat = function (x) {
    var relTime = baja.RelTime.make(x.getTime());
    return analyticsUtil.formatHHMM(relTime.getHoursPart(), relTime.getMinutesPart(), "chart.avgprofile.xaxis.fmt");
  };

  AverageProfileChart.prototype.getXAxisLabel = function () {
    return lex.get("chart.avgprofile.xaxis.label");
  };

  /**
   * Get the chart margins
   * @type {{top: number, right: number, bottom: number, left: number}}
   */
  AverageProfileChart.prototype.chartMargins = function () {
    return {
      top: 20,
      right: 50,
      bottom: 60,
      left: 70
    };
  };

  AverageProfileChart.prototype.processSeries = function (series) {
    var groupedSeries = _.groupBy(series, function (rec) {
      var time = rec.date;
      return time.getTime().getTimeOfDayMillis();
    });

    var ser = [];
    _.each(groupedSeries, function (value, key) {
      var finalTrendFlags = 0;
      var avg = d3.mean(value, function (d) {
        var dIntpStatus = d.trendFlags;
        finalTrendFlags = finalTrendFlags | dIntpStatus;
        return d.value;
      });
      var tf = analyticsUtil.decodeTrendFlag(finalTrendFlags);
      ser.push({ x: parseInt(key), y: avg, metaData: { interpolationStatus: tf } });
    });
    return ser;
  };

  /**
   * This needs a clean slate
   * @returns {boolean}
   */
  AverageProfileChart.prototype.needsCleanSlate = function () {
    return true;
  };

  /**
   *
   * @returns {boolean}
   */
  AverageProfileChart.prototype.needsBreakOnOneIteration = function () {
    return true;
  };

  /**
   * This method should be overridden by subclasses to return respective chart types.
   * The default implementation retruns "none".
   * @returns {string}
   */
  AverageProfileChart.prototype.getSupportedExportTypes = function () {
    return ["apchart", "csv"];
  };

  /**
   *
   * @returns {[string,string]}
   */
  AverageProfileChart.prototype.getColumnNames = function () {
    var arr = ['Time Of Day'],
        that = this,
        chartModelList = that.chartModelList;
    _.each(chartModelList, function (chartModel, index) {
      var sn = baja.SlotPath.unescape(chartModel.getSeriesName());
      if (_.isEmpty(chartModel.getUnitDisplay())) {
        arr.push(sn);
      } else {
        arr.push(lex.get({
          key: 'report.loadduration.colformat',
          args: [sn, chartModel.getUnitDisplay()]
        }));
      }
      arr.push(sn + " " + lex.get("chart.interpolationStatus.label"));
    });
    that.columnNames = arr;
    return arr;
  };

  /**
   *
   * @returns {[string,string]}
   */
  AverageProfileChart.prototype.getColumnKeys = function () {
    var arr = ['date'],
        that = this,
        chartModelList = that.chartModelList;
    _.each(chartModelList, function (chartModel, index) {
      var sn = chartModel.getSeriesName();
      arr.push(sn);
      arr.push(sn + "interpolationStatus");
    });
    that.columnKeys = arr;
    return arr;
  };

  /**
   *
   * @returns {*}
   */
  AverageProfileChart.prototype.getDefaultSettings = function () {
    var defObj = AnalyticC3BaseChart.prototype.getDefaultSettings.apply(this, arguments);
    defObj.chartType = "line";
    return defObj;
  };

  /**
   *
   * @param data
   * @param timeRange
   * @returns {*}
   */
  AverageProfileChart.prototype.buildModel = function (data, timeRange) {
    var model = AnalyticC3BaseChart.prototype.buildModel.apply(this, arguments);
    model.setShowOnY2(data.getShowOnY2());
    return model;
  };

  AverageProfileChart.prototype.getName = function () {
    return "AverageProfileChart";
  };

  return AverageProfileChart;
});
