/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/analyticsUtil
 */
define(['Promise', 'baja!', 'moment', 'underscore', 'nmodule/webChart/rc/webChartUtil', 'nmodule/webEditors/rc/servlets/registry'], function (Promise, baja, moment, _, webChartUtil) {
  "use strict";

  /**
   * A set of utility functions for analytics support within the webChart module. This utility is temporary until analytics has its
   * own analytics-ux module.
   * @exports nmodule/webChart/rc/analyticsUtil
   */

  var analyticsUtil = {},
      TREND_SCHEME = "analytictrend";

  /**
   * If this is an analytics trend ord scheme, attempt to match up the timeRange in the ord with the Model.
   * When the model is not yet initialized, the model will take its Time Range from the ord.
   * @param {module:nmodule/webChart/rc/model/BaseModel} model
   * @param {module:nmodule/webChart/rc/model/BaseSeries} series
   * @param {baja.Ord} ord
   * @param {baja.TimeZone} timeZone
   * @returns {Promise}
   */
  analyticsUtil.matchPreferredTimeRange = function (model, series, ord, timeZone) {
    if (!analyticsUtil.isAnalyticTrendOrdScheme(ord) || typeof model.timeRange !== "function") {
      return Promise.resolve();
    }

    return analyticsUtil.getPreferredTimeRangeFromOrd(model, ord, timeZone).then(function (prefTimeRange) {
      if (!prefTimeRange) {
        return;
      }

      if (!model.isInitialized() && !model.$customTimeRangeSet && !model.settings().isFile()) {
        //prevent double customization
        model.$customTimeRangeSet = true;
        model.timeRange(prefTimeRange); //set the model time range to the ord
        return analyticsUtil.modifyPreferredTimeRangeInOrd(model, prefTimeRange, ord, timeZone).then(function (newOrd) {
          series.$dataOrd = newOrd;
        });
      }

      //when already initialized, set the data ord to match the model's timeRange when its different
      var modelTimeRange = model.timeRange();
      if (!prefTimeRange.equivalent(modelTimeRange)) {
        return analyticsUtil.modifyPreferredTimeRangeInOrd(model, modelTimeRange, ord, timeZone).then(function (newOrd) {
          series.$dataOrd = newOrd;
        });
      }
    });
  };

  /**
   * Obtain the WebChartTimeRange from an AnalyticsEncodedTimeRange
   * @param {String} encoding
   * @return  {Promise.<baja.Struct>} resolves to `webChart:WebChartTimeRange`
   */
  analyticsUtil.getAnalyticsTimeRange = function (encoding) {
    return webChartUtil.rpc("type:analytics:AnalyticsRpc", "getStartEndTime", encoding).then(function (result) {

      return baja.$('webChart:WebChartTimeRange', {
        period: baja.$('webChart:WebChartTimeRangeType').make(1),
        startTime: webChartUtil.getAbsTime(moment(result.startTime)),
        endTime: webChartUtil.getAbsTime(moment(result.endTime))
      });
    });
  };

  /**
   * Obtain the encoding for the AnalyticsEncodedTimeRange based on WebChartTimeRange
   * @param {webChart.WebChartTimeRange}
   * @param {baja.TimeZone} timeZone
   * @return  {Promise.<String>} the encoding
   */
  analyticsUtil.getAnalyticsTimeRangeEncoding = function (timeRange, timeZone) {
    var results = webChartUtil.getStartAndEndDateFromTimeRange(timeRange, timeZone),
        startTime = webChartUtil.getAbsTime(results.start),
        endTime = webChartUtil.getAbsTime(results.end);

    return webChartUtil.rpc("type:analytics:AnalyticsRpc", "getAnalyticTimeRangeEncoding", startTime.getMillis(), endTime.getMillis()).then(function (result) {
      return result;
    });
  };

  /**
   * Return true if the ord has a AnalyticTrendOrdScheme
   *
   * @param  {String} ord The ORD to be resolved.
   * @returns {Boolean} Returns true if it is
   */
  analyticsUtil.isAnalyticTrendOrdScheme = function (ord) {
    var queries = baja.Ord.make(ord).parse(),
        isAnalyticTrend = false,
        schemeName,
        i;

    // Look for the analytictrend
    for (i = 0; i < queries.size(); ++i) {
      schemeName = queries.get(i).getSchemeName();
      if (schemeName.toLowerCase() === TREND_SCHEME) {
        isAnalyticTrend = true;
      }
    }

    return isAnalyticTrend;
  };
  /**
   * analytictrend provides no access to the Path, so remove that part of the query
   * when attempting to get the navDisplayOrd
   *
   * @param  {String} ord The ORD to be resolved.
   * @returns {baja.Ord} Returns the ord that can be used to get to a displayOrd
   */
  analyticsUtil.navDisplayOrd = function (ord) {
    var queries = baja.Ord.make(ord).parse(),
        schemeName,
        i;

    // Look for the analytictrend
    for (i = 0; i < queries.size(); ++i) {
      schemeName = queries.get(i).getSchemeName();
      if (schemeName.toLowerCase() === TREND_SCHEME) {
        queries.remove(i);
      }
    }

    return baja.Ord.make(queries);
  };

  /**
   * Return a Time Range if the ord has a preferred time range
   *
   * @param {module:nmodule/webChart/rc/model/BaseModel} model
   * @param  {String} ord The Ord to be checked.
   * @param {baja.TimeZone} timeZone
   * @returns {Promise.<webChart.WebChartTimeRange>} Resolves a promise to non-null TimeRange if time is preferred
   */
  analyticsUtil.getPreferredTimeRangeFromOrd = function (model, ord, timeZone) {
    var queries = baja.Ord.make(ord).parse(),
        schemeName,
        timeRange,
        i,
        j;

    // look for any analytictrend ORD schemes.
    for (i = 0; i < queries.size(); i++) {
      schemeName = queries.get(i).getSchemeName();
      if (schemeName.toLowerCase() === TREND_SCHEME) {
        var body = queries.get(i).getBody(),
            split = body.split("&");
        for (j = 0; j < split.length; j++) {
          var split2 = split[j].split("="),
              key = split2[0],
              value = split2[1];
          if (key === "timeRange") {
            try {
              // If loaded from dashboard and model is not initialized, then take tr from model.
              if (analyticsUtil.hasDashboardData(model) && !model.isInitialized()) {
                var flag = analyticsUtil.loadSettingsFromJson(model);
                value = flag.model().timeRange().getPeriod().getTag();
              }
              var timeRangeType = baja.$('webChart:WebChartTimeRangeType', value),
                  results = webChartUtil.getStartAndEndDateFromPeriod(timeRangeType, timeZone),
                  startTime = webChartUtil.getAbsTime(results.start),
                  endTime = webChartUtil.getAbsTime(results.end);

              timeRange = baja.$('webChart:WebChartTimeRange', {
                period: timeRangeType,
                startTime: startTime,
                endTime: endTime
              });
            } catch (err) {
              //analytics time Range must be using an analytics encoding
              webChartUtil.trace(err);
              return analyticsUtil.getAnalyticsTimeRange(value);
            }

            return Promise.resolve(timeRange);
          }
        }
      }
    }

    return Promise.resolve();
  };

  /**
   * attempt to modify the ord to update the Time Range if the ord has a preferred time range
   *
   * @param {module:nmodule/webChart/rc/model/BaseModel} model
   * @param {webChart.WebChartTimeChange} timeRange
   * @param  {String} ord The Ord to be checked.
   * @param {baja.TimeZone} timeZone
   * @returns {baja.Ord} Resolves a promise to non-null ord if its supposed to be modified
   */
  analyticsUtil.modifyPreferredTimeRangeInOrd = function (model, timeRange, ord, timeZone) {
    var queries = baja.Ord.make(ord).parse(),
        schemeName,
        i,
        j,
        valueToReplace;

    // modify any view ORD schemes.
    for (i = 0; i < queries.size(); i++) {
      schemeName = queries.get(i).getSchemeName();
      if (schemeName === TREND_SCHEME) {
        var body = queries.get(i).getBody();
        var split = body.split("&");
        for (j = 0; j < split.length; j++) {
          var split2 = split[j].split("=");
          var key = split2[0];
          var value = split2[1];
          if (key === "timeRange") {
            valueToReplace = value;
          }
        }
      }
    }

    if (!valueToReplace) {
      return Promise.resolve(ord);
    }

    return analyticsUtil.getAnalyticsTimeRangeEncoding(timeRange, timeZone).then(function (newEncoding) {
      var original = "timeRange=" + valueToReplace,
          replacement = "timeRange=" + newEncoding,
          newOrd = ord.replace(original, replacement);

      return newOrd;
    });
  };

  analyticsUtil.hasDashboardData = function (model) {
    return !_.isEmpty(model.settings().$widget.properties().getValue("dashboardChartData", false));
  };

  analyticsUtil.loadSettingsFromJson = function (model) {
    var widget = model.settings().$widget;
    var chartJson = JSON.parse(model.settings().$widget.properties().getValue("dashboardChartData", false));
    if (chartJson) {
      var settings = widget.settings();
      settings.loadFromJson(chartJson);
      widget.model().loadFromJson(chartJson, settings);

      if (chartJson.title) {
        widget.title(chartJson.title);
      }
    }
    return widget;
  };

  return analyticsUtil;
});
