/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Scott Hoye
 * @since Niagara 4.0
 */

/*jshint browser:true */
/*global niagara */

/**
 * API Status: **Private**
 * @module nmodule/search/rc/DefaultSearchResultsWidget
 */
define(['bajaux/Widget',
        'bajaux/dragdrop/dragDropUtils',
        'jquery',
        'Promise',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/webEditors/rc/util/htmlUtils',
        'nmodule/webEditors/rc/wb/menu/menuUtils',
        'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport',
        'nmodule/webEditors/rc/wb/commands/DeleteCommand',
        'hbs!nmodule/search/rc/template/DefaultSearchResultsWidget-content',
        'baja!',
        'baja!baja:IStatus,baja:Status,control:NumericPoint,control:BooleanPoint,control:EnumPoint,search:ResultsRequest',
        'lex!baja,search',
        'css!nmodule/search/rc/DefaultSearchResultsWidget'], function (
        Widget,
        dragDropUtils,
        $,
        Promise,
        fe,
        feDialogs,
        htmlUtils,
        menuUtils,
        addContextMenuSupport,
        DeleteCommand, // TODO: Find better way to filter right click commands
        tplContent,
        baja,
        types,
        lexs) {

  'use strict';

  var bajaLex = lexs[0],
      lex = lexs[1],
      TABLE_CLASS = '.n4-search-widget-table',
      RESULT_CLASS = '.n4-search-widget-result',
      DESCRIPTOR_CLASS = '.n4-search-widget-descriptor',
      DISPLAY_NAME_CLASS = '.n4-search-widget-name',
      ORD_CLASS = '.n4-search-widget-ord',
      STATUS_CLASS = '.n4-search-widget-status',
      VALUE_CLASS = '.n4-search-widget-value',
      SHORT_VALUE_CLASS = '.n4-search-widget-value-short',
      COMMANDS_CLASS = '.n4-search-widget-value-commands-div',
      POPOUT_CLASS = '.n4-search-widget-pop-button',
      HISTORY_CLASS = '.n4-search-widget-history-button',
      DISPLAY_NAME_TAG = baja.SlotPath.escape('n:displayName'),
      HISTORY_ID_TAG = baja.SlotPath.escape('n:history'),
      TYPE_TAG = baja.SlotPath.escape('n:type'),
      HISTORY_CHART_TXT = lex.get('SearchWidget.commands.historyChart'),
      LIVE_CHART_TXT = lex.get('SearchWidget.commands.liveChart'),
      HISTORY_TXT = lex.get('SearchWidget.commands.history'),
      LOADING_TXT = lex.get('SearchWidget.loading'),
      EDIT_TXT = lex.get('SearchWidget.commands.edit'),
      UNRESOLVED_TXT = lex.get('SearchWidget.unresolved.value'),
      EXPAND_TXT = lex.get('SearchWidget.commands.expand'),
      LAST_SELECT_CLS = 'lastSelected',
      LAST_SELECT_SELECTOR = '.' + LAST_SELECT_CLS,
      SELECT_CLS = 'selected',
      SELECT_SELECTOR = '.' + SELECT_CLS,
      CONTEXT_MENU_SELECTOR = '.contextMenu',

      contextMenuOnLongPress = htmlUtils.contextMenuOnLongPress,

      NEW_COMMAND_GROUP_DISPLAY_NAME = menuUtils.NEW_COMMAND_GROUP_DISPLAY_NAME;

  /**
   * The Default Search Results Widget.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/search/rc/DefaultSearchResultsWidget
   */
  var DefaultSearchResultsWidget = function DefaultSearchResultsWidget() {
    /** remember to call super constructor. Javascript won't do this for you */
    Widget.apply(this, arguments);
    addContextMenuSupport(this);

    // Override the $toContextMenu mixin function so that we can filter the commands
    // presented on a right click.
    // TODO: Revisit this approach for filtering right click commands, as it is fragile and as
    // new commands are added, we have to remember to modify this filter if needed.
    var _$toContextMenu = this.$toContextMenu;
    this.$toContextMenu = function () {
      return _$toContextMenu.apply(this, arguments)
        .then(function (commandGroups) {
          var size = (commandGroups)?commandGroups.size(): 0,
              i;

          // Remove the 'New' command group
          for (i = 0; i < size; i++ ) {
            if (commandGroups.get(i).getDisplayNameFormat() === NEW_COMMAND_GROUP_DISPLAY_NAME) {
              commandGroups.remove(i);
              break;
            }
          }

          // Remove the 'Delete' command
          return commandGroups.filter({include: function (cmd) {
            return !(cmd instanceof DeleteCommand);
          }});
        });
    };
  };

  //extend and set up prototype chain
  DefaultSearchResultsWidget.prototype = Object.create(Widget.prototype);
  DefaultSearchResultsWidget.prototype.constructor = DefaultSearchResultsWidget;

  /**
   * Do initial setup of the DOM for the Widget. This will set up the DOM's
   * structure and create a space where the commands and table will go.
   *
   * @param {jQuery} element the DOM element into which to load this Widget
   */
  DefaultSearchResultsWidget.prototype.doInitialize = function (dom) {
    var that = this;

    return new Promise(function (resolve, reject) {
      require(['hbs!nmodule/search/rc/template/DefaultSearchResultsWidget-result'],
        function (template) {
          that.$resultTemplate = template;
          resolve(that.initializeCommonSearchElements(dom));
        },
        reject
      );
    });
  };

  /**
   * Do initial setup of the DOM for the Widget. This will set up the DOM's
   * structure and create a space where the commands and table will go.
   *
   * @param {jQuery} element the DOM element into which to load this Widget
   */
  DefaultSearchResultsWidget.prototype.initializeCommonSearchElements = function (dom) {
    // This function is called when a search result row is clicked or dragged so that
    // the table selection can be properly updated
    function updateRowSelection(row, event) {
      var rowOrd = row.data('ord'),
          allRows = dom.find(RESULT_CLASS),
          lastSelectedRow = allRows.filter(LAST_SELECT_SELECTOR),
          rightClick = (event.which === 3),// TODO: If right click menu can ever handle multiple selections, remove this
          startIndex,
          endIndex,
          diff;

      if (!rightClick && event.shiftKey && (lastSelectedRow.length === 1)) {
        // Determine the start and end indexes of the rows to select, and also the number of rows
        // to select
        startIndex = lastSelectedRow.index();
        endIndex = row.index();
        if (endIndex < startIndex) {
          var temp = startIndex;
          startIndex = endIndex;
          endIndex = temp;
        }
        diff = endIndex - startIndex + 1;

        // Now add the selected class to the appropriate rows
        if (startIndex > 0) {
          startIndex = startIndex - 1;
          allRows = allRows.filter(':gt('+startIndex+')');
        }
        allRows = allRows.filter(':lt('+diff+')');
        allRows.toggleClass(SELECT_CLS, true);
      } else {
        allRows.each(function (i) {
          var kid = $(this),
            ord = kid.data('ord');

          if (!rightClick && event.ctrlKey) {
            if (ord === rowOrd) {
              kid.toggleClass(SELECT_CLS);
            }
          } else {
            kid.toggleClass(SELECT_CLS, ord === rowOrd);
          }
        });
      }

      // Keep track of the last selected row, as we'll need to know that for
      // the shift-click selection case in the future
      if (lastSelectedRow) {
        lastSelectedRow.removeClass(LAST_SELECT_CLS);
      }
      row.addClass(LAST_SELECT_CLS);
    }

    // Drag and Drop support of selected search result rows
    dom.on('dragstart', RESULT_CLASS, function (ev) {
      var row = $(this),
          selectedRows,
          json = [];

      if (!row.hasClass(SELECT_CLS)) {
        updateRowSelection(row, ev);
      }

      selectedRows = dom.find(RESULT_CLASS).filter(SELECT_SELECTOR);
      json = selectedRows.map(function () {
        var selectedRow = $(this),
            objName = selectedRow.data('name'),
            objType = selectedRow.data('type');
        return {
          ord: selectedRow.data('ord'),
          name: (objName)?objName:'',
          displayName: selectedRow.find(DISPLAY_NAME_CLASS).first().text(),
          description: '',
          typeSpec: (objType)?objType:''
        };
      }).get();

      // Start a drag operation.
      if (typeof niagara !== 'undefined' && niagara.env && niagara.env.startDrag) {
        //TODO: maybe dragDropUtils.startDrag. if i start a drag, why shouldn't workbench know about it?
        return niagara.env.startDrag(JSON.stringify({
          mime: 'niagara/navnodes',
          data: json
        }), ev);
      }
      else {
        dragDropUtils.toClipboard(ev.originalEvent.dataTransfer, 'niagara/navnodes', json);
      }
    });

    contextMenuOnLongPress(dom, {selector: CONTEXT_MENU_SELECTOR});

    // Allow multi selection of search result rows in the table
    // Use mouseup instead of mousedown, because mousedown will update the selection
    // before the dragging begins (ie. it would prevent dragging multiple selected results)
    dom.on('mouseup', RESULT_CLASS, function (ev) {
      if (ev.which !== 3) {
        updateRowSelection($(this), ev);
      }
    });

    dom.on('mousedown', RESULT_CLASS, function (ev) {
      if (ev.which === 3) {
        updateRowSelection($(this), ev);
      }
    });

    dom.on('contextmenu', RESULT_CLASS, function (ev) {
      if (ev.which === 3) {
        updateRowSelection($(this), ev);
      }
    });

    // Hyperlink to the selected result when user clicks on the display name
    dom.on('click', DISPLAY_NAME_CLASS, function () {
      var ord = $(this).data('ord');

      if (typeof niagara !== 'undefined' && niagara.env) {
        niagara.env.hyperlink(ord);
      }

      return false;
    });

    // Popout button handler
    dom.on('click', POPOUT_CLASS, function () {
      var ordStr = $(this).data('ord'),
          ord = baja.Ord.make(ordStr);

      ord.get().then(function(obj) {
        feDialogs.showFor({
          title: obj.getNavDisplayName(),
          value: obj,
          facets: obj.getFacets(),
          formFactor: Widget.formfactor.compact
        });
      });
    });

    // History button handler
    dom.on('click', HISTORY_CLASS, function () {
      var historyOrd = $(this).data('history'),
        ord = baja.Ord.make(historyOrd);

      if (typeof niagara !== 'undefined' && niagara.env) {
        niagara.env.hyperlink(ord);
      }
    });
  };

  /**
   * Loads in a BSearchService value and initializes the result subscriber.
   */
  DefaultSearchResultsWidget.prototype.doLoad = function (searchService) {
    initSubscriber(this, /*forceUnsubscribe*/true);
  };

  /**
   * Called by `destroy` so a developer has a chance to clean up
   * before the DOM is destroyed.
   *
   * @see module:bajaux/Widget#destroy
   *
   * @param {Object} [params] Optional params object passed to
   * `destroy()`
   * @param {$.Promise} An optional promise that's resolved once the widget has been destroyed.
   */
  DefaultSearchResultsWidget.prototype.doDestroy = function (params) {
    cleanupSubscriber(this, /*forceUnsubscribe*/true);
  };

  /**
   * This function builds a page of results for the last submitted search query
   *
   * @param searchService - This is the SearchService from which to retrieve search results
   * @param taskOrd - This parameter specifies the ORD to the submitted search task
   * @param subscribe - a {Boolean} value indicating whether or not to subscribe to results
   * @param pageIndex - The current page (zero based) in view // TODO: This will go away with infinite scrolling
   * @param pageSize - The page size // TODO: This will go away with infinite scrolling
   * @returns {Promise}
   */
  DefaultSearchResultsWidget.prototype.loadSearchResults = function (searchService,
                                                                     taskOrd,
                                                                     subscribe,
                                                                     pageIndex,
                                                                     pageSize) {
    var that = this,
        sub = that.$resultSubscriber,
        dom = that.jq(),
        resultsRequest = baja.$("search:ResultsRequest"),
        startIndex = pageIndex * pageSize;

    resultsRequest.setTaskOrd(taskOrd);
    resultsRequest.setStartIndex(startIndex);
    resultsRequest.setMaxResults(pageSize);

    // Invoke an action on the station's SearchService to retrieve the page of results
    return searchService.retrieveResults(resultsRequest)
      .then(function(resultSet) {
        dom.html(tplContent());
        return addResultRows(that, resultSet);
      })
      .then(function(ordArray) {
        if (sub && !sub.isEmpty()) {
          // Before we unsubscribe from the last page of results, we wait for
          // 60 seconds to give the effect of a subscription "linger".
          // Since subscription can be expensive to initiate and
          // there's a good chance the user may go back and forth between result pages,
          // this "linger" will save us some subscription time to make page loading
          // appear more responsive
          setTimeout(function() {
            sub.unsubscribeAll();
          }, 60000);
        }

        initSubscriber(that, /*forceUnsubscribe*/false);

        // Now we must resolve (and optionally subscribe) to each search result ORD
        // so that we can asynchronously update the table row values
        if (ordArray.length) {
          resolveSearchResults(that, ordArray, subscribe).catch(function (err) {});
        }
      })
      .catch(function(err) {
        dom.html('');
        throw err;
      });
  };

  /**
   * This function retrieves new search results from the search service and appends them
   * to the view.
   *
   * @param searchService - This is the SearchService from which to retrieve new search results
   * @param taskOrd - This parameter specifies the ORD to the submitted search task
   * @param subscribe - a {Boolean} value indicating whether or not to subscribe to results
   * @param startIndex - The starting index to use when retrieving search results
   * @param maxResults - The max number of search results to retrieve
   * @returns {Promise}
   */
  DefaultSearchResultsWidget.prototype.appendNewResults = function (searchService,
                                                                    taskOrd,
                                                                    subscribe,
                                                                    startIndex,
                                                                    maxResults) {
    // TODO: Need to refactor this function, as there is some code duplication between this
    // function and the loadSearchResults() function (with some subtle differences).  However, it might
    // become a moot point if/when we support infinite scrolling of results.
    var that = this,
        resultsRequest = baja.$("search:ResultsRequest");

    resultsRequest.setTaskOrd(taskOrd);
    resultsRequest.setStartIndex(startIndex);
    resultsRequest.setMaxResults(maxResults);

    // Invoke an action on the station's SearchService to retrieve the new results
    return searchService.retrieveResults(resultsRequest)
      .then(function(resultSet) {
        return addResultRows(that, resultSet);
      })
      .then(function(ordArray) {
        // Now we must resolve (and optionally subscribe) to each search result ORD
        // so that we can asynchronously update the table row values
        if (ordArray.length) {
          return resolveSearchResults(that, ordArray, subscribe).catch(function (err) {});
        }
      });
  };

  /**
   * Subscribe or unsubscribe from the search results currently in view
   *
   * @param subscribe a {Boolean} indicating whether or not to subscribe the search results
   * @returns {Promise}
   */
  DefaultSearchResultsWidget.prototype.updateSubscription = function (subscribe) {
    var that = this,
        dom = this.jq(),
        searchResults = dom.find(RESULT_CLASS),
        ordsToSubscribe = [],
        i;

    if (subscribe) {
      for (i = 0; i < searchResults.length; i++ ) {
        ordsToSubscribe.push(searchResults.eq(i).data('ord'));
      }
      return resolveSearchResults(that, ordsToSubscribe, subscribe).catch(function (err) {});
    } else {
      initSubscriber(that, /*forceUnsubscribe*/true);
      return Promise.resolve();
    }
  };

  /**
   * Returns the number of search results currently displayed for the current page
   * in view
   */
  DefaultSearchResultsWidget.prototype.getDisplayedResultsCount = function () {
    // TODO: This function may become obsolete when we have infinite scrolling
    return this.jq().find(RESULT_CLASS).length;
  };

  /**
   * Returns the handlebars template to use for a search result
   */
  DefaultSearchResultsWidget.prototype.getResultTemplate = function () {
    return this.$resultTemplate;
  };

  /**
   * Called when a user right clicks on search results, this function
   * returns a Promise which resolves the selected results.
   */
  DefaultSearchResultsWidget.prototype.getSubject = function () {
    // Find all selected result ORDs, resolve them asynchronously, and if they are components,
    // return them as an array and resolve the promise returned.
    var selectedRows = this.jq().find(RESULT_CLASS).filter(SELECT_SELECTOR),
        ords = [],
        batchResolve;

    if (selectedRows && (selectedRows.length)) {
      ords = selectedRows.map(function () {
        return $(this).data('ord');
      }).get();

      batchResolve = new baja.BatchResolve(ords);
      return batchResolve.resolve()
        .then(function () {
          return batchResolve.getTargetObjects();
        });
    } else {
      return Promise.resolve();
    }
  };

////////////////////////////////////////////////////////////////
// Private functions
////////////////////////////////////////////////////////////////

  /**
   * This private function initializes a new subscriber for the given
   * DefaultSearchResultsWidget after cleaning up the old one.
   *
   * @inner
   * @param widget - The DefaultSearchResultsWidget
   * @param forceUnsubscribe is a boolean flag that when true causes the
   * old subscriber to force an unsubscribeAll before dereferencing it
   */
  function initSubscriber(widget, forceUnsubscribe) {
    cleanupSubscriber(widget, forceUnsubscribe);

    widget.$resultSubscriber = new baja.Subscriber();

    // Attach the subscriber to listen for events on the search results
    widget.$resultSubscriber.attach("changed componentRenamed", function (prop, cx) {
      updateResult(widget, this, prop, cx);
    });
  }

  /**
   * This private function cleans up the old subscriber for the given
   * DefaultSearchResultsWidget by asynchronously unsubscribing from everything.
   *
   * @inner
   * @param widget - The DefaultSearchResultsWidget
   * @param forceUnsubscribe is a boolean flag that when true causes the
   * old subscriber to force an unsubscribeAll before dereferencing it
   */
  function cleanupSubscriber(widget, forceUnsubscribe) {
    if (widget.$resultSubscriber) {
      if (forceUnsubscribe) {
        widget.$resultSubscriber.unsubscribeAll();
      }
      widget.$resultSubscriber = null;
    }
  }

  /**
   * This private function processes search results and adds rows in the
   * table for each of the results.
   *
   * @inner
   * @param widget The DefaultSearchResultsWidget
   * @param searchResults A {search:SearchResultSet} instance containing the search
   * results
   * @returns {Promise} that when resolves passes an argument containing an Array
   * of {baja.Ord} containing the session ORDs for all search results added
   */
  function addResultRows(widget, searchResults) {
    var results = searchResults.getResults(),
        ordArray = [],
        dom = widget.jq(),
        resultTemplate = widget.getResultTemplate(),
        resultTable = dom.find(TABLE_CLASS).first();

    return importUnknownTypes(results).catch(function (err) {})
      .then(function() {
        // Add placeholder table rows for each search result found, but we
        // won't have actual values until later when we resolve/subscribe
        // to each search result
        results.getSlots().each(function (slot) {
          var result = results.get(slot),
            resultOrd = result.getOrd(),
            resultOrdStr = resultOrd.toString(),
            dispName = result.get(DISPLAY_NAME_TAG),
            typeSpec = result.get(TYPE_TAG),
            type,
            historyId = result.get(HISTORY_ID_TAG),
            historyOrd,
            historyTxt,
            historyImg,
            existing = resultTable.find(RESULT_CLASS).filter('[data-ord=\'' + resultOrdStr + '\']');

          // Avoid appending duplicate results, so check to make sure this search result
          // is not already displayed
          if (!(existing && (existing.length))) {
            if (dispName) {
              dispName = dispName.toString();
            } else {
              dispName = lex.get('SearchWidget.loadingName', slot.getName());
            }

            if (typeSpec) {
              type = baja.lt(typeSpec);
            }

            if (type &&
              (type.is('control:NumericPoint') ||
                type.is('control:BooleanPoint') ||
                type.is('control:EnumPoint') ||
                (type.is("niagaraVirtual:NiagaraVirtualControlPoint") &&
                 !type.is("niagaraVirtual:NiagaraVirtualStringPoint")))) {
              if (historyId) {
                historyOrd = baja.Ord.make('history:' + historyId.toString());
                historyTxt = HISTORY_CHART_TXT;
                historyImg = '/module/icons/x16/charts/line.png';
              } else {
                historyOrd = baja.Ord.make(resultOrdStr + '|view:webChart:ChartWidget');
                historyTxt = LIVE_CHART_TXT;
                historyImg = '/module/icons/x16/startHistory.png';
              }
            } else if (historyId) {
              historyOrd = baja.Ord.make('history:' + historyId.toString());
              historyTxt = HISTORY_TXT;
              historyImg = '/module/icons/x16/history.png';
            }

            resultTable.append(resultTemplate({
              displayName: dispName,
              ord: resultOrdStr,
              uri: resultOrd.toUri(),
              typeSpec: typeSpec,
              value: LOADING_TXT,
              popoutText: EDIT_TXT,
              history: historyOrd,
              historyText: historyTxt,
              historyImage: historyImg,
              expandText: EXPAND_TXT
            }));

            ordArray.push(resultOrd);

            // Only show the Popout (Edit) command button if there is a compact field
            // editor registered on the result type
            existing = resultTable.find(POPOUT_CLASS).filter('[data-ord=\'' + resultOrdStr + '\']');
            if (type) {
              fe.getDefaultConstructor(type, { formFactor: 'compact' })
                .then(function (Ed) {
                  if (!!Ed) {
                    existing.show();
                  } else {
                    if (!historyOrd) { // If no history either, hide the whole commands div
                      existing = resultTable.find(COMMANDS_CLASS).filter('[data-ord=\'' + resultOrdStr + '\']');
                    }
                    existing.hide();
                  }
                });
            } else {
              if (!historyOrd) { // If no history either, hide the whole commands div
                existing = resultTable.find(COMMANDS_CLASS).filter('[data-ord=\'' + resultOrdStr + '\']');
              }
              existing.hide();
            }
          }
        });

        // When the result table is modified, we need to call getDisplayedResultsCount() so
        // that the compact version can show/hide the expand all/collapse all button
        widget.getDisplayedResultsCount();

        return ordArray;
      });
  }

  /**
   * This private function updates a single row in the table of
   * results, and/or if the Niagara object being updated is the search task
   * itself, provide additional information to the user.
   *
   * @inner
   * @param widget The DefaultSearchResultsWidget
   * @param obj A Niagara object (must be a BISpaceNode) whose state should
   * be reflected in the appropriate table row. If this Niagara object is
   * the search task itself, it will also update the result statistics.
   * @param prop (Optional) The property on the Niagara object (component)
   * that changed (used when the search results are subscribed).
   * @param cx (Optional) The context associated with the property change.
   */
  function updateResult(widget, obj, prop, cx) {
    var ordInSession = obj.getOrdInSession().toString(),
        ordSelector = '[data-ord=\'' + ordInSession + '\']',
        dom = widget.jq(),
        row = dom.find(RESULT_CLASS).filter(ordSelector),
        rowName,
        rowType,
        td,
        nameAnchor,
        ordDiv,
        displayName,
        navOrd,
        navOrdStr;

    if (row) {
      // Set the name and type data attributes on the result row for drag-n-drop use
      rowName = row.data('name');
      if (!rowName && obj.getName) {
        row.data('name', obj.getName());
      }
      rowType = row.data('type');
      if (!rowType && obj.getType) {
        row.data('type', obj.getType().toString());
      }

      // Update the table row's value if applicable
      td = row.find(VALUE_CLASS);
      if (td) {
        td.text(obj.toString());
      }

      td = row.find(SHORT_VALUE_CLASS);
      if (td) {
        if (obj.getType && (obj.getType().is('control:ControlPoint') ||
            obj.getType().is("niagaraVirtual:NiagaraVirtualControlPoint")) &&
            obj.get('out')) {
          td.text(obj.get('out').getDisplay('value'));
        } else {
          td.text(obj.toString());
        }
      }

      updateValueColor(widget, obj, ordInSession);

      // Update the table row's display name
      td = row.find(DESCRIPTOR_CLASS);
      if (td) {
        nameAnchor = td.find(DISPLAY_NAME_CLASS);
        displayName = obj.getNavDisplayName();
        if (displayName && (displayName !== nameAnchor.text())) {
          nameAnchor.text(displayName);
        }

        ordDiv = row.find(ORD_CLASS);
        navOrd = obj.getNavOrd();
        if (navOrd) {
          navOrdStr = navOrd.relativizeToSession().toString();
          if (navOrdStr !== ordDiv.text()) {
            ordDiv.text(navOrdStr);
            nameAnchor.data('ord', navOrdStr);
            nameAnchor.attr('href', navOrd.toUri());
          }
        }
      }
    }
  }

  /**
   * Given an array of ORDs to search results, this private function resolves them
   * in batch (and optionally subscribe to them) so that we can provide asynchronous
   * updates to the table of search results.
   *
   * @inner
   * @param widget The DefaultSearchResultsWidget
   * @param ords An array of ORDs to the search results.
   * @param subscribe a {Boolean} indicating whether or not to subscribe the search results
   * @returns {Promise}
   */
  function resolveSearchResults(widget, ords, subscribe) {
    var dom = widget.jq(),
        sub = widget.$resultSubscriber,
        batchResolve = new baja.BatchResolve(ords);

    return batchResolve.resolve({ subscriber: sub }).finally(function() {
      var i = 0,
          failure,
          ordSelector,
          td,
          historyOrd;

      for (i = 0; i < ords.length; i++) {
        if (batchResolve.isResolved(i)) {
          updateResult(widget, batchResolve.get(i), null, null);
        }
        else {
          failure = batchResolve.getFail(i);
          if (failure !== null) {
            ordSelector = '[data-ord=\'' + ords[i] + '\']';
            td = dom.find(VALUE_CLASS + ', ' + SHORT_VALUE_CLASS).filter(ordSelector);

            if (td) {
              td.text(UNRESOLVED_TXT);
            }

            td = dom.find(ORD_CLASS).filter(ordSelector);
            if (td) {
              td.text(ords[i].toString());
            }

            // If the unresolved object has a history, we can still keep
            // the history command as long as its not the 'live' chart only one
            td = dom.find(HISTORY_CLASS).filter(ordSelector);
            if (td) {
              historyOrd = td.data('history');
            } else {
              historyOrd = null;
            }

            if (historyOrd && (historyOrd.indexOf('history:') === 0)) {
              td = dom.find(POPOUT_CLASS).filter(ordSelector);
            } else {
              td = dom.find(COMMANDS_CLASS).filter(ordSelector);
            }

            if (td) {
              td.hide();
            }
          }
        }
      }

      if (!subscribe && !sub.isEmpty()) {
        // Even if we aren't subscribing to the results, we still need to "lease" them
        // just momentarily in order to retrieve the status so that status colors can be
        // shown. We can't use Component.lease because it requires all ORDs live in the same space,
        // so instead we enforce a manual temporary subscription.
        sub.unsubscribeAll();
      }
    });
  }

  /**
   * Given search results, determine if any types need to be imported asynchronously.
   *
   * @inner
   * @param searchResults The {baja:Component} instance containing search results as child slots
   * @returns {Promise}
   */
  function importUnknownTypes(searchResults) {
    var typeSpecsToImport = [];

    searchResults.getSlots().each(function (slot) {
      var typeSpec = searchResults.get(slot).get(TYPE_TAG),
          type;

      if (typeSpec) {
        type = baja.lt(typeSpec);
        if (!type) {
          typeSpecsToImport.push(typeSpec);
        }
      }
    });

    if (!baja.lt("niagaraVirtual:NiagaraVirtualComponent")) {
      typeSpecsToImport.push("niagaraVirtual:NiagaraVirtualComponent");
    }

    if (!baja.lt("niagaraVirtual:NiagaraVirtualControlPoint")) {
      typeSpecsToImport.push("niagaraVirtual:NiagaraVirtualControlPoint");
    }

    if (!baja.lt("niagaraVirtual:NiagaraVirtualStringPoint")) {
      typeSpecsToImport.push("niagaraVirtual:NiagaraVirtualStringPoint");
    }

    if (typeSpecsToImport.length > 0) {
      return baja.importTypes({
        typeSpecs: typeSpecsToImport
      }).catch(function (err) {});
    } else {
      return Promise.resolve();
    }
  }

  /**
   * Given a search result object, determine if it is a BIStatus value, and if so
   * update the value cell color appropriately to reflect the status.
   *
   * @inner
   * @param widget The DefaultSearchResultsWidget
   * @param obj The Niagara object (must be a BISpaceNode) whose state should
   * be reflected in the appropriate table row.
   * @param ordStr The Ord in Session for the Niagara Object as a {String}
   */
  function updateValueColor(widget, obj, ordStr) {
    if (obj &&
        typeof obj.getType === 'function' &&
        obj.getType().is('baja:IStatus')) {
      var status,
          fgColor = 'inherit',
          bgColor = 'transparent',
          ordSelector = '[data-ord=\'' + ordStr + '\']',
          td = widget.jq().find(STATUS_CLASS).filter(ordSelector);

      // Check for the non-control point Niagara Virtual use case.
      // This is useful to find the status property on Niagara Virtual devices, networks, etc.
      if (obj.getType().is("niagaraVirtual:NiagaraVirtualComponent") &&
          !obj.getType().is("niagaraVirtual:NiagaraVirtualControlPoint")) {
        var statusProp = obj.get("status");
        if (statusProp && statusProp.getType().is("baja:Status")) {
          status = statusProp;
        }
      }

      if (!status) {
        status = baja.Status.getStatusFromIStatus(obj);
      }

      if (td && td.length && status) {
        if (!widget.$statusFgColors) {
          // Lazy load the status foreground colors
          // Array index 0 = alarmFg
          // Array index 1 = disabledFg
          // Array index 2 = faultFg
          // Array index 3 = downFg
          // Array index 4 = overriddenFg
          // Array index 5 = staleFg
          widget.$statusFgColors = [convertColorCode(bajaLex.get('Status.alarm.fg')),
                                    convertColorCode(bajaLex.get('Status.disabled.fg')),
                                    convertColorCode(bajaLex.get('Status.fault.fg')),
                                    convertColorCode(bajaLex.get('Status.down.fg')),
                                    convertColorCode(bajaLex.get('Status.overridden.fg')),
                                    convertColorCode(bajaLex.get('Status.stale.fg'))];
        }
        if (!widget.$statusBgColors) {
          // Lazy load the status background colors
          // Array index 0 = alarmBg
          // Array index 1 = disabledBg
          // Array index 2 = faultBg
          // Array index 3 = downBg
          // Array index 4 = overriddenBg
          // Array index 5 = staleBg
          widget.$statusBgColors = [convertColorCode(bajaLex.get('Status.alarm.bg')),
                                    convertColorCode(bajaLex.get('Status.disabled.bg')),
                                    convertColorCode(bajaLex.get('Status.fault.bg')),
                                    convertColorCode(bajaLex.get('Status.down.bg')),
                                    convertColorCode(bajaLex.get('Status.overridden.bg')),
                                    convertColorCode(bajaLex.get('Status.stale.bg'))];
        }

        if (status.isDisabled()) {
          fgColor = widget.$statusFgColors[1];
          bgColor = widget.$statusBgColors[1];
        }
        else if (status.isFault()) {
          fgColor = widget.$statusFgColors[2];
          bgColor = widget.$statusBgColors[2];
        }
        else if (status.isDown()) {
          fgColor = widget.$statusFgColors[3];
          bgColor = widget.$statusBgColors[3];
        }
        else if (status.isAlarm()) {
          fgColor = widget.$statusFgColors[0];
          bgColor = widget.$statusBgColors[0];
        }
        else if (status.isStale()) {
          fgColor = widget.$statusFgColors[5];
          bgColor = widget.$statusBgColors[5];
        }
        else if (status.isOverridden()) {
          fgColor = widget.$statusFgColors[4];
          bgColor = widget.$statusBgColors[4];
        }

        // Set the color and background-color of the table cell
        td.css('color', fgColor);
        td.css('background-color', bgColor);
      }
    }
  }

  /**
   * Given a hex color code from the baja lexicon, convert it to an
   * appropriate CSS hex color code
   *
   * @inner
   * @param hexString The {String} form of the hex color code retrieved from the baja lexicon
   * @returns A {String} containing a hex color code appropriate for use in CSS
   */
  function convertColorCode(hexString) {
    if (hexString && hexString.length > 7) {
      hexString = hexString.replace('FF', '');
    }

    return hexString;
  }

  return DefaultSearchResultsWidget;
});
