/**
 * @file Time-related utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!'], function (baja) {

  "use strict";

  /** 
   * Number of milliseconds in one day 
   * @name niagara.util.time.MILLIS_IN_DAY 
   */

  var MILLIS_IN_DAY = 86400000,

  /**
   * Number of milliseconds in one hour
   * @name niagara.util.time.MILLIS_IN_HOUR
   */
  MILLIS_IN_HOUR = 3600000,

  /**
   * Number of milliseconds in half an hour
   * @name niagara.util.time.MILLIS_IN_HALF_HOUR
   */
  MILLIS_IN_HALF_HOUR = 1800000,

  /**
   * Number of milliseconds in one minute
   * @name niagara.util.time.MILLIS_IN_MINUTE
   */
  MILLIS_IN_MINUTE = 60000,

  /**
   * Number of milliseconds in one second
   * @name niagara.util.time.MILLIS_IN_SECOND
   */
  MILLIS_IN_SECOND = 1000;

  /**
   * Normalizes milliseconds past midnight into a baja.Time. Passing in a
   * negative number or more than MILLIS_IN_DAY will not throw an error like
   * baja.Time.make() will.
   * 
   * @memberOf niagara.util.time
   * @function
   * 
   * @param {Number} millis milliseconds past midnight
   * @return baja.Time corresponding time of day
   */
  function makeTime(millis) {
    millis = (millis + MILLIS_IN_DAY) % MILLIS_IN_DAY;
    return baja.Time.make(millis);
  }

  /**
   * For a Javascript `Date`, returns the number of milliseconds past midnight
   * (UTC). For a `baja.Time`, returns `getTimeOfDayMillis()`.
   * 
   * @memberOf niagara.util.time
   * @param {Date|baja.Time} time
   * @returns {Number} milliseconds past midnight
   */
  function millisOfDay(date) {
    if (date instanceof Date) {
      return date.getTime() % MILLIS_IN_DAY;
    } else if (date instanceof baja.Time) {
      return date.getTimeOfDayMillis();
    } else if (typeof date === 'number') {
      return makeTime(date).getTimeOfDayMillis();
    }
  }

  /**
   * @memberOf niagara.util.time
   * @param {Date|baja.Time} time
   * @returns {Number} the number of milliseconds past the hour.
   */
  function millisOfHour(date) {
    if (date instanceof Date) {
      return date.getTime() % MILLIS_IN_HOUR;
    } else if (date instanceof baja.Time) {
      return date.getTimeOfDayMillis() % MILLIS_IN_HOUR;
    } else if (typeof date === 'number') {
      return makeTime(date).getTimeOfDayMillis() % MILLIS_IN_HOUR;
    }
  }

  /**
   * Rounds the given `Date` or `baja.Time` to the nearest half hour.
   * 
   * @memberOf niagara.util.time
   * @param {Date|baja.Time} time
   * @returns {Date|baja.Time} the input time, rounded to the nearest half
   * hour
   */
  function roundToHalfHour(date) {
    var moh = millisOfDay(date),
        rounded = Math.round(moh / MILLIS_IN_HALF_HOUR) * MILLIS_IN_HALF_HOUR;

    if (date instanceof Date) {
      return new Date(rounded);
    } else if (date instanceof baja.Time) {
      return baja.Time.make({ relTime: baja.RelTime.make(rounded) });
    } else if (typeof date === 'number') {
      return rounded;
    }
  }

  function zeroPad(num) {
    return (num < 10 ? '0' : '') + num;
  }

  function toTimeString(hours, minutes, seconds) {
    var arr = [zeroPad(hours), zeroPad(minutes), zeroPad(seconds || 0)];
    return arr.join(':');
  }

  /** 
   * @namespace
   * @name niagara.util.time
   */
  return {
    MILLIS_IN_DAY: MILLIS_IN_DAY,
    MILLIS_IN_HOUR: MILLIS_IN_HOUR,
    MILLIS_IN_HALF_HOUR: MILLIS_IN_HALF_HOUR,
    MILLIS_IN_MINUTE: MILLIS_IN_MINUTE,
    MILLIS_IN_SECOND: MILLIS_IN_SECOND,
    makeTime: makeTime,
    millisOfDay: millisOfDay,
    millisOfHour: millisOfHour,
    roundToHalfHour: roundToHalfHour,
    toTimeString: toTimeString,
    zeroPad: zeroPad
  };
});
