/**
 * @file Slot-related utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'jquery', 'Promise', 'underscore', 'baja!baja:Link,baja:Password,control:IWritablePoint'], function (baja, $, Promise, _) {

  "use strict";

  function arrayShift(array, srcIndex, destIndex) {
    if (srcIndex === destIndex || srcIndex < 0 || destIndex < 0 || srcIndex >= array.length || destIndex >= array.length) {
      return false;
    }

    var removed = array.splice(srcIndex, 1);
    array.splice(destIndex, 0, removed[0]);
    return true;
  }

  function doMoveSlot(component, prop, obj, doSwap) {
    var slotArray = component.getSlots().dynamic().toArray(),
        index = _.indexOf(slotArray, component.getSlot(prop));
    if (doSwap(slotArray, index)) {
      return component.reorder($.extend(obj, {
        dynamicProperties: slotArray
      }));
    } else {
      return Promise.resolve();
    }
  }

  /**
   * Reorders the component's dynamic slots so that the given property
   * moves up one in the list.
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Component} component the component whose slots to reorder
   * @param {baja.Slot} slot the slot to move
   * @returns {Promise}
   */
  function moveUp(component, slot, obj) {
    return doMoveSlot(component, slot, obj, function (array, index) {
      return arrayShift(array, index, index - 1);
    });
  }

  /**
   * Reorders the component's dynamic slots so that the given property
   * moves down one in the list.
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Component} component the component whose slots to reorder
   * @param {baja.Slot} slot the slot to move 
   * @returns {Promise}
   */
  function moveDown(component, slot, obj) {
    return doMoveSlot(component, slot, obj, function (array, index) {
      return arrayShift(array, index, index + 1);
    });
  }

  /**
   * Reorders the component's dynamic slots so that the given property
   * moves to the top of the list.
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Component} component the component whose slots to reorder
   * @param {baja.Slot} slot the slot to move 
   * @returns {Promise}
   */
  function moveToTop(component, slot, obj) {
    return doMoveSlot(component, slot, obj, function (array, index) {
      return arrayShift(array, index, 0);
    });
  }

  /**
   * Reorders the component's dynamic slots so that the given property
   * moves to the bottom of the list.
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Component} component the component whose slots to reorder
   * @param {baja.Slot} slot the slot to move 
   * @returns {Promise}
   */
  function moveToBottom(component, slot, obj) {
    return doMoveSlot(component, slot, obj, function (array, index) {
      return arrayShift(array, index, array.length - 1);
    });
  }

  /**
   * Check to see if we need to pop up a confirmation dialog before firing
   * this action.
   * 
   * @memberOf niagara.util.slot
   * 
   * @param {baja.Action} action the action we want to fire
   * @return true if we require confirmation before firing this action
   */
  function isConfirmRequired(action) {
    var flags = action.getFlags();
    if (baja.Flags.CONFIRM_REQUIRED & flags) {
      return true;
    }

    return false;
  }

  /**
   * Check to see if a slot has the READONLY flag set.
   * 
   * @memberOf niagara.util.slot
   * 
   * @param {baja.Slot} slot
   * @return true if the READONLY flag is set
   */
  function isReadonly(slot) {
    if (slot.getType().is('baja:Password')) {
      return true;
    }

    return baja.Flags.READONLY & slot.getFlags();
  }

  /**
   * Check to see if this slot should be hidden from display on the property
   * sheet.
   * 
   * @memberOf niagara.util.slot
   * 
   * @param {baja.Slot} slot property to check for hidden status
   * @returns {Boolean} true if property should be hidden
   */
  function isHidden(slot) {
    var flags = slot.getFlags();

    if (baja.Flags.HIDDEN & flags) {
      return true;
    } else if (slot.isProperty() && slot.getType().is('baja:Link')) {
      return true;
    } else if (slot.getName() === 'wsAnnotation' || slot.getName() === 'displayNames') {
      return true;
    }

    return false;
  }

  /**
   * Check to see if clicking this property in the property sheet should link
   * to a field editor. To be editable, a property must not be readonly or
   * hidden.
   * 
   * @memberOf niagara.util.slot
   * 
   * @param {baja.Property} prop property to check for editability
   * @returns {Boolean} true if we should link to a field editor
   */
  function isEditable(prop) {
    if (prop.getType().is('baja:Password')) {
      return false;
    } else if (isReadonly(prop)) {
      return false;
    } else if (isHidden(prop)) {
      return false;
    }

    return true;
  }

  /**
   * Check to see if an action can be fired (if the slot is an action, and
   * the slot is not hidden).
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Slot} slot
   * @returns {Boolean} if the slot is an action that can be fired
   */
  function isFireable(slot) {
    return slot.isAction() && !isHidden(slot);
  }

  /**
   * Attempt to pull Facets off of a Complex. First attempt to pull facets
   * directly from the Complex using `baja.Complex#getFacets`. If the Complex
   * has a frozen slot of type `baja:Facets`, pull those off and merge those 
   * in - in the case of duplicate Facets, facets from the frozen slot will win
   * out.
   * 
   * @memberOf niagara.util.slot
   * @private
   * @param {baja.Complex} complex
   * @return {baja.Facets}
   */
  function getComplexFacets(complex) {
    var facets = baja.Facets.DEFAULT,
        complexFacets,
        frozenSlotFacets;
    if (complex && complex.getType().isComplex()) {
      frozenSlotFacets = complex.getSlots().properties().frozen().is('baja:Facets').firstValue();

      if (typeof complex.getFacets === 'function') {
        complexFacets = complex.getFacets();
        if (complexFacets) {
          facets = baja.Facets.make(facets, complexFacets);
        }
      }

      if (frozenSlotFacets) {
        facets = baja.Facets.make(facets, frozenSlotFacets);
      }
    }

    return facets;
  }

  /**
   * Finds the nearest ancestor to the given Complex that is of type
   * Component (this may be the input Complex itself).
   * 
   * @memberOf niagara.util.slot
   * @private
   * @param {baja.Complex} comp
   */
  function getComponentAncestor(comp) {
    var type = comp && comp.getType();
    if (type) {
      if (type.isComponent()) {
        return comp;
      } else if (type.isComplex()) {
        return getComponentAncestor(comp.getParent());
      }
    } else {
      return undefined;
    }
  }

  /**
   * Attempts to retrieve a usable Facets object from a particular Baja
   * value. Station-side, there is a significant amount of specialized
   * facets-retrieving logic (often in the form of overrides of
   * `getSlotFacets()` - since this code isn't available to Bajascript, we'll
   * have to replicate the most relevant bits as best we can in a way that
   * makes sense for our mobile apps.
   * 
   * @memberOf niagara.util.slot
   * @param {baja.Value} value a value to examine for facet information
   * @param {baja.Slot|String} [slot] a slot to examine for facet information
   * @returns {baja.Facets} any facet information that could be retrieved
   * from the given value and/or slot - or `baja.Facets.DEFAULT` if no facets
   * were found
   */
  function getFacets(value, slot) {
    var facets = baja.Facets.DEFAULT,
        type = value && value.getType(),
        ancestor,
        ancestorFacets,
        ancestorType,
        slotName;

    // did we pass in a Slot as well as a value? If so, see if this value
    // is a Complex and the slot references a Complex as well - if so, grab
    // the Complex property referenced by the slot and examine that one.
    if (slot && value.getType().isComplex()) {
      slot = value.getSlot(slot);
      if (slot.isProperty() && slot.getType().isComplex()) {
        value = value.get(slot);
      }
    }

    if (type) {
      if (type.isComplex()) {
        // walk up the component tree until we find a Component
        ancestor = getComponentAncestor(value);
        if (ancestor) {
          ancestorType = ancestor.getType();
          ancestorFacets = getComplexFacets(ancestor);
          //merge our Component facets in with our facets so far
          if (ancestorFacets) {
            facets = baja.Facets.make(facets, ancestorFacets);
          }
        }
      }
    }

    // now see if the slot itself has facets configured
    if (slot instanceof baja.Slot) {
      slotName = String(slot);
      if (ancestorType && ancestorType.is('control:IWritablePoint')) {
        if (slot.isFrozen() && slotName.indexOf('in') === 0 || slotName === 'set' || slotName === 'fallback' || slotName === 'override' || slotName === 'emergencyOverride') {
          //these action slots cannot have their own facets configured -
          //we will use the contents of the 'facets' slot instead, which
          //comes from the calls to getComplexFacets above.
          //see BEnumWritable#getSlotFacets()
          return facets;
        }
      }
      // merge slot facets in with our facets so far
      facets = baja.Facets.make(facets, slot.getFacets());
    }

    return facets;
  }

  /**
   * @namespace
   * @name niagara.util.slot
   */
  return {
    getFacets: getFacets,
    isConfirmRequired: isConfirmRequired,
    isEditable: isEditable,
    isFireable: isFireable,
    isHidden: isHidden,
    isReadonly: isReadonly,
    moveDown: moveDown,
    moveUp: moveUp,
    moveToTop: moveToTop,
    moveToBottom: moveToBottom
  };
});
