/**
 * @file Niagara Mobile Utils - functions relating to the
 * "available views" button and associated dialog in our JQM page headers
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*global niagara*/

define(['baja!', 'lex!mobile', 'Promise', 'bajaux/commands/Command', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/mobile', 'mobile/util/mobile/commands'], function (baja, lexs, Promise, Command, dialogs, mobileUtil, commands) {

  "use strict";

  var mobileLex = lexs[0];

  /**
   * Finds out which of the available views can be linked to. A view will only
   * be linked to if it is not the same as the current view.
   * 
   * @memberOf niagara.util.mobile.views
   * @private
   * @param {baja.Component} list the list of all available views retrieved
   * from a `ComponentServerSideHandler.getMobileAgentList()` server-side call
   * @return {Array} an array of linkable view objects - each object has
   * `displayName` and `url` properties - may be empty if no available linkable
   * views
   */
  function getLinkableViews(list) {
    var slots = list.getSlots(),
        linkables = [];

    baja.iterate(slots, function (slot) {
      var prop = list.get(slot);

      linkables.push({
        displayName: prop.get('displayName'),
        url: prop.get('ord').toUri()
      });
    });

    return linkables;
  }

  /**
   * Assembles the list of alternate views for a component into a listview
   * dialog. If no alternate views are available, an OK dialog saying so will 
   * be shown instead.
   * 
   * @memberOf niagara.util.mobile.views
   * @private
   * @param {baja.Component} list the list of all available views retrieved
   * from a `ComponentServerSideHandler.getMobileAgentList()` server-side call
   * @return {Object} a dialog handler for the shown dialog
   */
  function showListDialog(list, callbacks) {
    var views = getLinkableViews(list),
        viewCommands = [];
    if (views.length) {
      baja.iterate(views, function (view) {
        viewCommands.push(new Command(view.displayName, function () {
          mobileUtil.linkToOrd(view.url);
        }));
      });
      commands.showCommandsDialog(viewCommands, undefined, callbacks);
    } else {
      return dialogs.ok(mobileLex.get('views.noneAvailable'));
    }
  }

  /**
   * Shows a listview dialog with other views available for the specified ORD.
   * The entries in the listview will link to alternate views/apps for the given
   * ORD (for example, when viewing an alarm console recipient component in the 
   * Alarm Console app, can switch to view it in the Property Sheet.
   * 
   * @memberOf niagara.util.mobile.views
   * @param {baja.Ord|String|baja.Component} [ord] the ORD we wish to find 
   * alternate views for. You can also pass a mounted Component here - this
   * saves a network call to the station to resolve the ORD, and the alternate
   * views will be found for that component. If undefined/null, 
   * `niagara.view.ord` will be used.
   * @returns {Promise} promise to be resolved once the dialog has been
   * created and displayed
   */
  function showViewsDialog(ord) {

    ord = ord || niagara.view.ord;

    mobileUtil.showPageLoadingMsg();

    return baja.rpc({
      typeSpec: "mobile:ComponentServerSideHandler",
      method: "getMobileAgentList",
      args: [String(ord)]
    }).then(function (result) {
      return baja.bson.decodeValue(JSON.parse(result));
    }).then(function (list) {
      //finally, assemble them into a listview
      showListDialog(list);
    }).catch(dialogs.error).finally(function () {
      mobileUtil.hidePageLoadingMsg();
    });
  }

  // View Command for the Commands dialog
  var viewsCmdOrd = "",
      viewsCmd = new Command("%lexicon(mobile:views.selectView)%", function () {
    // When the link is clicked, show the views dialog
    return showViewsDialog(viewsCmdOrd);
  });

  // Set the ORD for the views command
  viewsCmd.setOrd = function (ord) {
    viewsCmdOrd = ord;
  };

  /**
   * Return the option object for selecting views.
   *
   * @see niagara.util.mobile.commands   
   *
   * @memberOf niagara.util.mobile.views
   * @returns {Object}
   */
  function getViewsCommand() {
    return viewsCmd;
  }

  // Automatically add the views command if we can
  if (niagara.view.profile.showViews && commands) {
    commands.prependDefaultCommand(getViewsCommand());
  }

  /**
   * @namespace
   * @name niagara.util.mobile.views
   */
  return {
    getViewsCommand: getViewsCommand
  };
});
