/**
 * @file Niagara Mobile Utils - Pages
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 *@namespace An event framework for managing internal Pages in jQuery Mobile.
 */
define(['baja!', 'jquery', 'jquerymobile', 'mobile/util/mobile/mobile'], function (baja, $, jqm, mobileUtil) {

  // Use ECMAScript 5 Strict Mode
  "use strict";

  var first = true,
      prevPage = null,
      currentPage = null,
      reg = [],
      loading = false,
      encodePageId = mobileUtil.encodePageId,
      getPageContainer = mobileUtil.getPageContainer,
      onPageContainerCreate = mobileUtil.onPageContainerCreate,
      DEFAULT_PAGE = '____DEFAULT_PAGE_$_$';

  /**
   * Register a Page.
   * 
   * This registers a Page handler object with the pages framework. After an object is
   * registered, it will receive event notifications and callbacks.
   * 
   * For instance, if the page wanted to receive a 'pageshow' event, the handler object would
   * need a function called 'pageshow'.
   * 
   * Handler objects also support dynamic page creation. This is done by defining a 
   * 'createPage' method on the handler. This method returns a String, jQuery DOM or 
   * DOM element that's then injected into the page container. The method is passed
   * a jQuery Mobile URL object.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @param {String|Function} matcher a function or string for matching a page name.
   * @param {Object} handler an object that will attempt to have notification methods called upon it.
   */
  function register(matcher, handler) {
    if (arguments.length === 1) {
      handler = matcher;
      matcher = DEFAULT_PAGE;
    }

    var m = matcher;
    if (typeof m === "string") {
      m = function m(s) {
        return matcher === s;
      };
    }

    // Register page so it can be looked up
    reg.push({
      isMatch: m,
      handler: handler
    });

    return handler;
  }

  /**
   * Return the specified Page Handler object.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @param {String} pageName.
   * @return {Object} the page handler Object or null if the page can't be found.
   */
  function getHandler(pageName) {
    if (!pageName) {
      return null;
    }
    var i;
    for (i = 0; i < reg.length; ++i) {
      if (reg[i].isMatch(pageName)) {
        return reg[i].handler;
      }
    }
    return null;
  }

  /**
   * Fire an event to the specified registered Page Handler.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @private
   */
  function fire(obj) {
    //baja.outln(obj);
    var nm = obj.page.attr("id") || DEFAULT_PAGE,
        handler = getHandler(nm);

    if (handler && typeof handler[obj.eventName] === "function") {
      handler[obj.eventName](obj);
    }
  }

  /**
   * Fire an event to all the registered Page Handlers.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @param {Object} obj the Object Literal for the method's arguments.
   * @param {String} obj.eventName the name of the event to fire
   */
  function fireAll(obj) {
    obj = baja.objectify(obj, "eventName");
    var i;
    for (i = 0; i < reg.length; ++i) {
      if (typeof reg[i].handler[obj.eventName] === "function") {
        reg[i].handler[obj.eventName](obj);
      }
    }
  }

  /**
   * Return the previous Page's name.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {String}
   */
  function getPrevName() {
    return prevPage.attr("id") || "";
  }

  /**
   * Return the previous Page's handler.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {Object}
   */
  function getPrevHandler() {
    return getHandler(getPrevName());
  }

  /**
   * Return the previous page DOM object.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {Object}
   */
  function getPrev() {
    return prevPage;
  }

  /**
   * Return the current Page name.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {String}
   */
  function getCurrentName() {
    return currentPage.attr("id") || "";
  }

  /**
   * Return the current page DOM object.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {Object}
   */
  function getCurrent() {
    return currentPage;
  }

  /**
   * Return the current page handler object.
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {Object}
   */
  function getCurrentHandler() {
    return getHandler(getCurrentName());
  }

  /**
   * Is the current page the very first page being shown?
   *
   * @memberOf niagara.util.mobile.pages
   *
   * @return {Boolean}
   */
  function isFirst() {
    return first;
  }

  /*
  The jQuery Mobile event sequence
  
  On start up... 
  pagebeforeshow   Current Page: alarmConsole    Page Name: undefined Next: false
  pageshow         Current Page: alarmConsole    Page Name: undefined Next: false
   On page change...
  pagebeforehide   Current Page: alarmSource     Page Name: alarmSource Next: true
  pagebeforeshow   Current Page: alarmSource     Page Name: alarmConsole Next: false
  pagehide         Current Page: alarmSource     Page Name: alarmSource Next: true
  pageshow         Current Page: alarmSource     Page Name: alarmConsole Next: false
  */

  // Register for the events   

  function pageshow(event, ui) {
    var activePage = ui.toPage,
        pPage;

    if (ui && ui.prevPage) {
      pPage = $(ui.prevPage);
    }
    if (!prevPage) {
      pPage = activePage;
    }

    fire({
      page: activePage,
      prevPage: pPage,
      event: event,
      eventName: 'pageshow'
    });

    if (activePage.jqmData('role') !== 'dialog') {
      prevPage = currentPage;
      currentPage = activePage;
    }
  }

  function pagebeforeshow(event, ui) {
    var activePage = $(event.target),
        pPage;

    if (!activePage) {
      return;
    }

    if (ui && ui.prevPage) {
      pPage = $(ui.prevPage);
    }
    if (!prevPage) {
      pPage = activePage;
    }

    fire({
      page: activePage,
      prevPage: pPage,
      event: event,
      eventName: event.type
    });
  }

  function pagehide(event, ui) {
    var nextPage;
    if (ui && ui.nextPage) {
      nextPage = $(ui.nextPage);
    }

    fire({
      page: currentPage,
      nextPage: nextPage,
      event: event,
      eventName: 'pagehide'
    });
  }

  function pagebeforehide(event, ui) {
    first = false;
    var nextPage;
    if (ui && ui.nextPage) {
      nextPage = $(ui.nextPage);
    }

    fire({
      page: currentPage,
      nextPage: nextPage,
      event: event,
      eventName: event.type
    });
  }

  function genericFire(event, ui) {
    fire({
      page: $(event.target),
      event: event,
      eventName: event.type
    });
  }

  //commented out due to JQM 1.0 wonkiness
  //https://github.com/jquery/jquery-mobile/issues/142#issuecomment-2983197
  //  var swipeTicket = baja.clock.expiredTicket;
  //  $("body").on("swiperight swipeleft", "div", function (event, ui) {
  //    // Throttle the swipe events into one
  //    if (swipeTicket.isExpired()) {
  //      swipeTicket = baja.clock.schedule(function () {
  //        fire({
  //          page: currentPage, 
  //          eventName: event.type
  //        });
  //      }, 20);
  //    }
  //  });

  function normalizeOrdUrl(url) {
    url = decodeURI(url);
    if (url.charAt(0) === '/' || url.charAt(0) === '#') {
      url = url.substring(1);
    }
    if (url.match(/^ord[\?\/]/)) {
      return 'ord?' + encodeURI(url.substring(4));
    } else {
      return encodeURI(url);
    }
  }

  function toId(path) {
    return encodePageId(normalizeOrdUrl(path));
  }

  function createPageDom(handler, id, options) {
    if (!handler || typeof handler.createPage !== "function") {
      throw new Error("No page handler (or no createPage function) " + "found for id " + id);
    }

    handler.createPage(options);
  }

  function load(path, options) {
    if (loading) {
      return;
    }

    var handler = getHandler(path),
        pageContainer = getPageContainer(),
        id;

    loading = true;

    if (options.pageData && handler && typeof handler.encodeUrl === "function") {
      path = handler.encodeUrl(options.pageData);
    }

    id = toId(path);

    if (!pageContainer.children("#" + id).length) {
      createPageDom(handler, id, $.extend(options, {
        ok: function ok(page) {
          page.attr('id', id);
          pageContainer.append(page);
          page.jqmData('pageData', options.pageData);
          page.page();
          loading = false;
          $.mobile.changePage(page, options);
        },
        fail: function fail(err) {
          loading = false;
          baja.fail(err);
        }
      }));
    }
  }

  function pagebeforechange(e, data) {
    // We only want to handle changePage() calls where the caller is asking us to load a page by URL.
    if (typeof data.toPage !== "string") {
      if (data.toPage instanceof $) {
        /*
        http://api.jquerymobile.com/pagebeforechange/
        this gets triggered twice when called programmatically. hooray
         */
        if (data.consumed) {
          return;
        }
        data.consumed = true;

        //if we are going from one JQM page to another (i.e. no dialogs
        //involved), let the pages handler have a shot at doing its own
        //pagebeforechange event.
        if (data.toPage.jqmData('role') !== 'dialog') {
          if (data.options.fromPage && data.options.fromPage.jqmData('role') !== 'dialog') {
            fire({
              page: data.options.fromPage,
              nextPage: data.toPage,
              event: e,
              eventName: 'pagebeforechange',
              options: data.options
            });
          }

          if (!e.isDefaultPrevented()) {
            setTimeout(function () {
              data.toPage.trigger('pagelayout');
            }, 0);
          }
        }
      }
      return;
    }

    // Attempt to decode page information from the URL    
    var url = $.mobile.path.parseUrl(data.toPage),
        href = normalizeOrdUrl(url.pathname + url.search),
        hash = normalizeOrdUrl(url.hash || ""),
        pathname = normalizeOrdUrl(url.pathname || ""),
        pageName = hash,
        pageContainer = getPageContainer(),
        handler,
        pageDom,
        pageData;

    // If we detect a dialog in the hash then bail
    if (hash.indexOf("&ui-state=dialog") > -1) {
      return;
    }

    // Get the page handler
    handler = getHandler(pageName);
    if (!handler) {
      pageName = pathname;
      handler = getHandler(pageName);
    }
    if (!handler) {
      pageName = href;
      handler = getHandler(pageName);
    }

    // If there's no page handler then bail
    if (!handler) {
      return;
    }

    // Look up the page name via its id
    pageDom = pageContainer.children("#" + toId(pageName));
    if (!pageDom.length && hash) {
      pageDom = pageContainer.children("#" + toId(hash));
    }
    if (!pageDom.length) {
      pageDom = pageContainer.children("#" + toId(href));
    }

    if (!hash || !pageDom.length) {
      /*
       * if we're intercepting a pagechange to an ORD so we can dynamically
       * inject the necessary page, then force dataUrl to the ord we want.
       * otherwise (e.g. we are doing hash-based navigation to a static page),
       * we want to keep the dataUrl JQM gives us, so don't.
       */
      data.options.dataUrl = '/' + href;
    }

    if (!pageDom.length) {
      if (typeof handler.createPage === "function") {
        pageData = handler.decodeUrl && handler.decodeUrl(url);
        data.options.pageData = pageData;
        data.options.url = url;

        /*
         * only do the dynamic page load AFTER baja starts because it may
         * (will) need to perform network calls to the station
         */
        baja.started(function () {
          load(pageName, data.options);
        });
      } else {
        // If the handler doesn't support dynamic page creation then bail
        return;
      }
    }

    // If a dialog then bail (currently we only support views)
    if (data.options.role && data.options.role === "dialog" || pageDom.is(".ui-dialog")) {
      return;
    }

    if (pageDom.length) {
      // Change to this new page
      $.mobile.changePage(pageDom, data.options);
    }

    // Prevent default change handler
    e.preventDefault();
  }

  function bindJQMEvents() {
    var pageContainer = getPageContainer(),
        pageSelector = 'div';

    pageContainer.on('pagebeforecreate pagecreate pageinit pagelayout', pageSelector, genericFire).on('pagecontainershow', pageshow).on('pagebeforeshow', pageSelector, pagebeforeshow).on('pagecontainerhide', pagehide).on('pagebeforehide', pageSelector, pagebeforehide);

    pageContainer.on("pagecontainerbeforechange", pagebeforechange);
  }

  onPageContainerCreate(bindJQMEvents);

  // Functions exported as public
  return {
    register: register,
    getPrevName: getPrevName,
    getPrev: getPrev,
    getPrevHandler: getPrevHandler,
    getCurrentName: getCurrentName,
    getCurrent: getCurrent,
    getCurrentHandler: getCurrentHandler,
    getHandler: getHandler,
    fire: fire,
    fireAll: fireAll,
    isFirst: isFirst
  };
});
