/**
 * @file Functions relating to day editing (block resizing, dragging,
 * etc.) within the mobile scheduler app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'jquery', 'mobile/util/time', 'mobile/schedule/util.schedule', 'mobile/util/jquery/enclosingPoint'], function (baja, $, timeUtil, scheduleUtil) {

  "use strict";

  /**
   * Functions relating to day editing (block resizing, dragging,
   * etc.) within the mobile scheduler app.
   * 
   * @private
   * @exports mobile/schedule/schedule.ui.day
   */

  var exports = {};

  var DAY = timeUtil.MILLIS_IN_DAY,
      HALF_HOUR = timeUtil.MILLIS_IN_HALF_HOUR,
      selectedScheduleBlockDiv;

  /**
   * Records the start of a mouse drag to create, resize, or move a schedule
   * block.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we've clicked on
   * @param {JQuery} div the div where the day lives
   * @param {JQuery} scheduleBlockDiv the schedule block div, if any, that
   * we have clicked on (if we have one, we're doing a move or resize; if not,
   * we're doing a create)
   * @param {String} operation one of `create, move, resizeTop, resizeBottom`
   * @param {number} pageY the pageY of the jQuery click event
   */
  function startDrag(day, div, scheduleBlockDiv, operation, pageY) {
    var start,
        finish,
        scheduleBlock = scheduleBlockDiv.data('scheduleBlock');

    if (scheduleBlock) {
      scheduleBlockDiv.addClass('selected');
      div.trigger('selectionchange');
      start = scheduleBlock.start;
      finish = scheduleBlock.finish;
    } else {
      start = timeUtil.makeTime(scheduleUtil.pixelsToMillis(div, pageY - div.offset().top));
      finish = timeUtil.makeTime(start.getTimeOfDayMillis() + HALF_HOUR);
    }

    day.dragStart = {
      pageY: pageY,
      timeStamp: baja.clock.ticks(),
      scheduleBlock: scheduleBlock,
      start: start,
      finish: finish,
      operation: operation
    };

    selectedScheduleBlockDiv = scheduleBlockDiv;
  }

  /**
   * Given a mouse click, find the divs where the click lies within that div's
   * bounds.
   * 
   * @param {JQuery} div only find divs that are children of this parent div
   * @param {JQuery.Event} event the mouse click event
   * @return {JQuery} the divs that lie under this mouse click
   */
  function getDivsUnderMouse(div, event) {
    return div.find(':not(.overlay)').filter(':enclosingPoint(' + event.pageX + ',' + event.pageY + ')');
  }

  /**
   * Attempts to set new start/finish times on a schedule block.
   * 
   * @memberOf niagara.schedule.ui.day
   * @private
   * @param {module:mobile/schedule/DayEditor} day the day editor
   * containing this block
   * @param {module:mobile/schedule/ScheduleBlock} block the block to change
   * start/stop times on
   * @param {baja.Time} startTime
   * @param {baja.Time} finishTime
   * @throws {Error} if finish time before start time, or if the new times
   * overlap with an existing block on the given day editor
   */
  function setStartStop(day, block, startTime, finishTime) {
    var startms = timeUtil.millisOfDay(startTime),
        stopms = timeUtil.millisOfDay(finishTime),
        existingStart = block.start,
        existingFinish = block.finish;

    if (startTime && startTime.equals(existingStart) && finishTime && finishTime.equals(existingFinish)) {
      return; //no action necessary
    }

    if (stopms > 0 && startms >= stopms) {
      throw "finish time (" + finishTime + ") " + "was before start time (" + startTime + ")";
    }

    if (!day.canMoveBlockTo(block, startms, stopms)) {
      throw "(" + startTime + "), (" + finishTime + ") overlaps with another block";
    }

    block.start = startTime;
    block.finish = finishTime;
  }

  /**
   * Moves a schedule block in response to a mouse drag. Handles both move
   * and resize operations.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we are moving a schedule block around in
   * @param {JQuery} div the div where the day lives
   * @param {Number} delta how far, in pixels, we have dragged the mouse down
   */
  function doTimeChange(day, div, delta) {
    /*
     * on a slower device, if you start dragging around before the page has
     * finished laying itself out then you might not have a selected block -
     * hence no time to change
     */
    if (!selectedScheduleBlockDiv || !selectedScheduleBlockDiv.data('scheduleBlock')) {
      return;
    }

    var dragStart = day.dragStart,
        operation = dragStart.operation,
        millis = scheduleUtil.pixelsToMillis(div, delta),
        dragDirection = millis > 0 ? 'down' : 'up',
        scheduleBlock = dragStart.scheduleBlock,
        start = dragStart.start.getTimeOfDayMillis(),
        finish = dragStart.finish.getTimeOfDayMillis() || DAY,
        newStart = start,
        newFinish = finish,
        setSuccessfully = false,
        exhaustedMillis = false;

    while (!setSuccessfully && !exhaustedMillis) {
      try {
        switch (operation) {
          //resizing top - move the top border, but don't let it go 
          //earlier than midnight
          case 'resizeTop':
            newStart = Math.max(start + millis, 0);
            newStart = Math.min(newStart, (newFinish || DAY) - HALF_HOUR);
            break;
          //resizing bottom - move the bottom border, but don't let it
          //go later than midnight
          case 'resizeBottom':
            newFinish = Math.min(finish + millis, DAY);
            newFinish = Math.max(newFinish, newStart + HALF_HOUR);
            break;
          //dragging the whole block - start and finish must move as one - if we
          //overlap midnight, setStartStop will throw an exception
          case 'move':
            newStart = start + millis;
            newFinish = finish + millis;
            break;
        }
        newStart = timeUtil.roundToHalfHour(newStart);
        newFinish = timeUtil.roundToHalfHour(newFinish);
        //try and set the new times on the block
        setStartStop(day, scheduleBlock, timeUtil.makeTime(newStart), timeUtil.makeTime(newFinish));
        setSuccessfully = true;
      } catch (failedToSet) {
        //could not set - we either crossed the midnight border during a drag,
        //or we overlapped another block. bump down the amount we're moving
        //and try again until we reach 0
        switch (dragDirection) {
          case 'down':
            millis -= HALF_HOUR;
            exhaustedMillis = millis <= 0;
            break;
          case 'up':
            millis += HALF_HOUR;
            exhaustedMillis = millis >= 0;
            break;
        }
      }
    }

    if (setSuccessfully) {
      dragStart.changed = true;
      scheduleUtil.updateScheduleBlockDiv(selectedScheduleBlockDiv);
      day.setModified(true);
    }
  }

  /**
   * Creates a new schedule block in response to a mouse drag.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we're creating a new schedule block in
   * @param {JQuery} div the div where the day lives
   * @param {Number} delta how far, in pixels, we have dragged the mouse down
   */
  function doCreate(day, div, delta) {
    var dragStart = day.dragStart,
        startPixels = dragStart.pageY - div.children('.blocksDisplay').offset().top,
        start = timeUtil.roundToHalfHour(scheduleUtil.pixelsToMillis(div, startPixels)),
        defaultValue = scheduleUtil.getCurrentSchedule().getNewBlockValue(),
        scheduleBlock = day.addBlock(timeUtil.makeTime(start), timeUtil.makeTime(start + 60000), defaultValue),
        blockDiv = scheduleBlock.generateDiv(div.children('.blocksDisplay'));

    dragStart.changed = true;

    startDrag(day, div, blockDiv, "resizeBottom", dragStart.pageY);
    doTimeChange(day, div, delta);
  }

  /**
   * Given a set of divs underneath our mouse click, decides which one is
   * determined to be "on top" - top/bottom resizers always win, followed
   * by the schedule block itself, the block divs on the day itself coming
   * in last.
   * 
   * @param {JQuery} divs the divs from whom to pick a winner
   */
  function getPriorityDiv(divs) {
    var pDiv;

    pDiv = divs.filter('.grab');
    if (pDiv.length) {
      return pDiv;
    }
    pDiv = divs.filter('.scheduleBlockContainer');
    if (pDiv.length) {
      return pDiv;
    }
    return divs.filter('.block');
  }

  /**
   * When we receive a mousedown on a grabber/resize div/divs, we must decide
   * which grabber "wins."
   * 
   * Often two schedule blocks abut each other, or one block is small enough
   * that the top/bottom grabber divs overlap each other. In this case, we
   * examine the total screen range occupied by the two divs and check to see
   * whether the click event occurred closer to the top or to the bottom. The
   * top or bottom grabber, respectively, then wins out over the other.
   * 
   * Of course if only one grabber lies under the mouse click, that div is
   * the de facto winner.
   * 
   * @param {JQuery} grabs the divs to choose a winner from (these divs should be
   * preselected as those with the `grab` CSS class)
   * @param {JQuery.Event} event the mousedown event
   * @returns {JQuery} the winning grabber div
   */
  function chooseGrabber(grabs, event) {
    if (grabs.length === 1) {
      return $(grabs[0]);
    }

    var minGrabTop = Number.MAX_VALUE,
        maxGrabBottom = Number.MIN_VALUE,
        minBlockTop = Number.MAX_VALUE,
        maxBlockBottom = Number.MIN_VALUE,
        highestGrab = null,
        lowestGrab = null;

    grabs.each(function () {
      var grab = $(this),
          grabTop = grab.offset().top,
          grabBottom = grabTop + grab.outerHeight(),
          block = grab.parent(),
          blockTop = block.offset().top,
          blockBottom = blockTop + block.outerHeight();

      if (grabTop < minGrabTop || grabTop === minGrabTop && blockTop < minBlockTop) {
        highestGrab = grab;
      }
      if (grabBottom > maxGrabBottom || grabBottom === maxGrabBottom && blockBottom > maxBlockBottom) {
        lowestGrab = grab;
      }
      minGrabTop = Math.min(grabTop, minGrabTop);
      maxGrabBottom = Math.max(grabBottom, maxGrabBottom);
      minBlockTop = Math.min(blockTop, minBlockTop);
      maxBlockBottom = Math.max(blockBottom, maxBlockBottom);
    });

    return event.pageY < (maxGrabBottom + minGrabTop) / 2 ? highestGrab : lowestGrab;
  }

  /**
   * Handle a mousedown on this day. Depending on what div we're clicking
   * on (a schedule block, a top/bottom resizer, the day itself), will 
   * initiate a mouse drag to perform a move, resize, or create operation.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we're clicking on
   * @param {JQuery} div the div where the day lives
   * @param {JQuery.Event} event the mousedown event
   */
  function handleMousedown(day, div, event) {
    var divs = getDivsUnderMouse(div, event),
        clickedDiv = getPriorityDiv(divs),
        scheduleBlockDiv,
        operation;

    $('.scheduleBlockContainer.selected', div).removeClass('selected');

    if (!clickedDiv.length) {
      return;
    } else if (clickedDiv.hasClass('scheduleBlockContainer')) {
      scheduleBlockDiv = clickedDiv;
      operation = 'move';
    } else if (clickedDiv.hasClass('grab')) {
      clickedDiv = chooseGrabber(clickedDiv, event);
      scheduleBlockDiv = clickedDiv.parent();
      operation = clickedDiv.hasClass('top') ? 'resizeTop' : 'resizeBottom';
    } else {
      scheduleBlockDiv = clickedDiv;
      operation = 'create';
    }

    div.trigger('selectionchange');

    startDrag(day, div, scheduleBlockDiv, operation, event.pageY);
  }

  /**
   * Handle a mousemove on this day. If we have previously clicked - i.e. we
   * are in the middle of a drag - will attempt to perform any appropriate
   * move/resize/create operations on the day's schedule blocks.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we're moving the mouse in
   * @param {JQuery} div the div where the day lives
   * @param {JQuery.Event} event the mousemove event
   */
  function handleMove(day, div, event) {

    var divs = getDivsUnderMouse(div, event),
        overlayDiv = div.find('.overlay'),
        dragStart = day.dragStart,
        delta;

    if (divs.filter('.grab').length) {
      overlayDiv.css('cursor', 'n-resize');
    } else {
      overlayDiv.css('cursor', 'default');
    }

    if (!dragStart) {
      return;
    }

    delta = event.pageY - dragStart.pageY;

    if (Math.abs(delta) > 20) {
      dragStart.moved = true;
    }

    if (dragStart.operation === "resizeTop" || dragStart.operation === "resizeBottom" || dragStart.operation === "move") {
      doTimeChange(day, div, delta);
    } else if (dragStart.operation === "create") {
      if (delta >= scheduleUtil.millisToPixels(div, HALF_HOUR)) {
        doCreate(day, div, delta);
      }
    }
  }

  /**
   * Handles a mouseup event on this day. Cancels any mouse drags in process.
   * If the mouse button was not held down too long (tap/hold) or moved too far
   * (drag), then a click event will be fired on any schedule block divs under
   * the mouse position.
   *
   * @param {module:mobile/schedule/DayEditor} day the day we let go of the mouse on
   * @param {JQuery} div the div where the day lives
   * @param {JQuery.Event} event the mouseup event
   */
  function handleMouseup(day, div, event) {
    var dragStart = day.dragStart,
        delta,
        timeDelta;

    if (!dragStart) {
      return;
    }

    delta = event.pageY - dragStart.pageY;
    timeDelta = baja.clock.ticks() - dragStart.timeStamp;

    //750 = JQM taphold timeout
    if (Math.abs(delta) < 3 && timeDelta < 750) {
      //click or tap - if we're on top of a schedule block, fire its click
      //event so we go to the editor for this schedule block
      getDivsUnderMouse(div, event).filter('.scheduleBlockContainer').click();
    }

    if (dragStart.changed) {
      day.setModified(true);
    }
    delete day.dragStart;
  }

  /**
   * Creates event handlers for a particular day editor.
   * 
   * Events for mousedown/mouseup/mousemove (and their corresponding touch
   * equivalents) are created for the editor's overlay div. mouseup/touchend
   * events are also bound on the document element (to handle the case where 
   * the user drags outside of the day editor.
   * 
   * These events will actually be bound to their corresponding DOM elements
   * in `armHandlers`.
   * 
   * @param {module:mobile/schedule/DayEditor} day the day we let go of the mouse on
   * @param {JQuery} div the div where the day lives
   */
  function createHandlers(day, div) {
    var throttledHandleMove = baja.throttle(handleMove, 50),
        supportsTouch = 'ontouchend' in document;

    if (supportsTouch) {
      return {
        document: {
          touchend: function touchend(event) {
            handleMouseup(day, div, event);
          }
        },
        overlay: {
          touchend: function touchend(event) {
            var touch = event.originalEvent.touches[0] || event.originalEvent.changedTouches[0];
            handleMouseup(day, div, touch);
          },
          touchmove: function touchmove(event) {
            var touch = event.originalEvent.touches[0] || event.originalEvent.changedTouches[0];
            throttledHandleMove(day, div, touch);
          },
          touchstart: function touchstart(event) {
            event.preventDefault();
            var touch = event.originalEvent.touches[0] || event.originalEvent.changedTouches[0];
            handleMousedown(day, div, touch);
          }
        }
      };
    } else {
      return {
        document: {
          mouseup: function mouseup(event) {
            handleMouseup(day, div, event);
          }
        },
        overlay: {
          mousedown: function mousedown(event) {
            handleMousedown(day, div, event);
          },
          mousemove: function mousemove(event) {
            throttledHandleMove(day, div, event);
          },
          mouseup: function mouseup(event) {
            handleMouseup(day, div, event);
          }
        }
      };
    }
  }

  /**
   * Arms all mousedown/move/up handlers on this day.
   * 
   * @memberOf niagara.schedule.ui.day
   * @function
   * 
   * @param {module:mobile/schedule/DayEditor} day the day to arm event listeners on
   * @param {JQuery} div the div where the day lives
   */
  exports.armHandlers = function armHandlers(day, div) {
    var overlayDiv = $('<div class="overlay"/>').appendTo(div.children('.blocksDisplay')),
        handlers = createHandlers(day, div);

    baja.iterate(handlers.overlay, function (handler, name) {
      overlayDiv.on(name, handler);
    });

    div.on('mousedown', handlers.overlay.mousedown);
    div.on('touchstart', handlers.overlay.touchstart);

    baja.iterate(handlers.document, function (handler, name) {
      $(document).on(name, handler);
    });

    day.unbindAll = function () {
      baja.iterate(handlers.overlay, function (handler, name) {
        overlayDiv.off(name, handler);
      });

      div.off('mousedown', handlers.overlay.mousedown);
      div.off('touchstart', handlers.overlay.touchstart);

      baja.iterate(handlers.document, function (handler, name) {
        $(document).off(name, handler);
      });
    };
  };

  return exports;
});
