var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Functions relating to the field editor for trigger times on
 * a `BTriggerSchedule`'s `times` slot (a `BDaySchedule`).
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!schedule:TimeSchedule,schedule:DaySchedule', 'baja!', 'lex!mobile,schedule', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/mobile', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/fieldeditors.mobile', 'mobile/fieldeditors/MobileFieldEditor', 'mobile/fieldeditors/mobile/feUtils', 'bajaux/events'], function (types, baja, lexs, $, jqm, Promise, _, dialogs, mobileUtil, fe, mobileFE, MobileFieldEditor, feUtils, events) {

  'use strict';

  var _lexs = _slicedToArray(lexs, 2),
      mobileLex = _lexs[0],
      scheduleLex = _lexs[1],
      getActivePage = mobileUtil.getActivePage,
      getBajaTimeFormat = feUtils.getBajaTimeFormat,
      TIME_SCHEDULE_TYPE = 'schedule:TimeSchedule',
      DAY_SCHEDULE_TYPE = 'schedule:DaySchedule';

  var checkboxHtml = '\n    <div class="checkboxContainer" data-role="fieldcontain">\n      <fieldset data-role="controlgroup" data-type="vertical">\n        <input type="checkbox" id="{checkboxId}"/>\n        <label for="{checkboxId}"><span>Selected</span></label>\n      </fieldset>\n    </div>';

  var triggerAddHtml = '\n    <div>\n      <label class="rangeStartLabel">{triggerTime}:</label>\n      <div class="triggerStartTime"></div>\n    </div>\n    <div>\n      <div class="triggerRange"></div>\n    </div>\n    <div class="triggerRangeEditors hidden">\n      <div>\n        <label class="rangeEndLabel">{rangeEnd}:</label>\n        <div class="triggerEndTime"></div>\n      </div>\n      <div>\n        <label class="rangeIntervalLabel">{rangeInterval}:</label>\n        <div class="triggerInterval"></div>\n      </div>\n      <div>\n        <label class="triggersToCreate"/>\n      </div>\n    </div>';

  /**
   * A field editor that will display a list of the triggers currently 
   * configured on a `schedule:TriggerSchedule`. Registered using key
   * `triggers-list`.
   * 
   * @inner
   * @class
   * @name TriggersListEditor
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */
  var TriggersListEditor = fe.defineEditor(MobileFieldEditor, /** @lends TriggersListEditor.prototype */{

    /**
     * When clicking a trigger's list item, check the checkbox.
     * 
     * @name niagara.fieldEditors.schedule.TriggersListEditor#doInitialize
     * @function
     * @param {jQuery} element
     * @see module:mobile/fieldeditors/MobileFieldEditor#doInitialize
     */
    doInitialize: function doInitialize(element) {
      element.on('click', 'li', function () {
        var checkbox = $(this).find('input[type=checkbox]'),
            checked = checkbox.is(':checked');
        checkbox.prop('checked', !checked).trigger('change').checkboxradio('refresh');
      });
    },

    /**
     * Assembles the triggers on a trigger schedule into a listview of the
     * trigger times with checkboxes to select each one.
     * 
     * @name niagara.fieldEditors.schedule.TriggersListEditor#doLoad
     * @function
     * @param {baja.Component} value a `schedule:DaySchedule` existing as the
     * `times` slot on a `schedule:TriggerSchedule`.
     * @see module:mobile/fieldeditors/MobileFieldEditor#doLoad
     */
    doLoad: function doLoad(value) {
      var that = this,
          dom = that.jq(),
          ul = $('<ul data-role="listview" data-inset="true" data-theme="c" />'),
          header = $('<li data-role="list-divider"/>'),
          timeSchedules;

      header.append(checkboxHtml.patternReplace({
        checkboxId: 'selectAll'
      })).append($('<div class="detailsContainer"/>').text(mobileLex.get('selectAll')));

      ul.append(header);

      timeSchedules = value.getSlots().properties().is(TIME_SCHEDULE_TYPE).toValueArray();

      // sort by time of day
      timeSchedules.sort(function (timeSchedA, timeSchedB) {
        var millisA = timeSchedA.getStart().getTimeOfDayMillis(),
            millisB = timeSchedB.getStart().getTimeOfDayMillis();
        return millisA - millisB;
      });

      // now assemble them into a listview
      baja.iterate(timeSchedules, function (timeSchedule) {
        var li = $('<li/>').appendTo(ul),
            checkboxContainer = $(checkboxHtml.patternReplace({
          checkboxId: timeSchedule.getName()
        })),
            detailsContainer = $('<div class="detailsContainer"/>');

        timeSchedule.getStart().toTimeString({
          textPattern: getBajaTimeFormat() || 'HH:mm',
          ok: function ok(timeStr) {
            detailsContainer.append(timeStr);
          }
        });

        li.append(checkboxContainer).append(detailsContainer);
      });

      // enhance and load into DOM
      dom.html(ul).trigger('create');
    },

    /**
     * Just returns `value()` - other functions on this editor will make
     * changes to the currently edited schedule as the user interacts with the
     * editor.
     * 
     * @name niagara.fieldEditors.schedule.TriggersListEditor#doRead
     * @function
     * @see module:mobile/fieldeditors/MobileFieldEditor#doRead
     */
    doRead: function doRead() {
      return this.value();
    }
  });

  /**
   * Returns the names of all the slots the user currently has selected
   * (checkboxes are checked).
   * 
   * @returns {Array} an array of slot names
   */
  TriggersListEditor.prototype.getSelectedSlots = function getSelectedSlots() {
    var slots = [];

    this.jq().find('.ui-li-static input[type=checkbox]').each(function () {
      var $this = $(this);
      if ($this.is(':checked')) {
        slots.push($this.attr('id'));
      }
    });

    return slots;
  };

  /**
   * Removes all the selected slots from the currently edited trigger schedule.
   * 
   * @returns {Promise}
   */
  TriggersListEditor.prototype.removeSelected = function removeSelected() {
    var that = this,
        component = that.value(),
        selectedSlots = that.getSelectedSlots();

    var removes = _.map(selectedSlots, function (slot) {
      return component.remove({ slot: slot });
    });

    return Promise.all(removes).then(function () {
      return that.load(component);
    });
  };

  fe.register(DAY_SCHEDULE_TYPE, TriggersListEditor, 'triggers-list');

  function toggleRangeDisplay(dom, isRange) {
    var method = isRange ? 'show' : 'hide',
        activePage = getActivePage();

    dom.find('.triggerRangeEditors').animate({
      height: method,
      opacity: method
    }, {
      step: function step(now, fx) {
        dialogs.repositionDialog(activePage);
      }
    });
  }

  function hasTime(daySchedule, time) {
    return daySchedule.getSlots().properties().is(TIME_SCHEDULE_TYPE).eachValue(function (timeSchedule) {
      return timeSchedule.getStart().equals(time);
    });
  }

  function doRange(startTime, endTime, interval) {
    var time = startTime,
        timeMillis = startTime.getTimeOfDayMillis(),
        timeSchedules = [],
        intervalMillis = interval.getMillis();

    if (intervalMillis < baja.RelTime.MILLIS_IN_MINUTE) {
      throw new Error(mobileLex.get('schedule.message.intervalRequired'));
    }

    function hasMoreTime() {
      return timeMillis < baja.RelTime.MILLIS_IN_DAY && endTime && !time.isAfter(endTime); //inclusive
    }

    do {
      timeSchedules.push(baja.$(TIME_SCHEDULE_TYPE, {
        start: time,
        finish: baja.Time.make(timeMillis + 20000) //+ 20secs
      }));

      timeMillis += intervalMillis;
      time = baja.Time.make(timeMillis);
    } while (hasMoreTime());

    return timeSchedules;
  }

  function readAddTriggerValues(triggersAddEditor) {
    var startEd = triggersAddEditor.startEditor,
        rangeEd = triggersAddEditor.rangeEditor,
        endEd = triggersAddEditor.endEditor,
        intervalEd = triggersAddEditor.intervalEditor;

    return Promise.join(startEd.read(), rangeEd.read(), endEd.read(), intervalEd.read()).spread(function (start, isRange, end, interval) {
      // if the user bumped hours up past 24, datebox will still record a day
      // even though the days spinner is disabled, so let's mod it off
      var intervalMillis = interval.getMillis(),
          newInterval = baja.RelTime.make(intervalMillis % baja.RelTime.MILLIS_IN_DAY);

      return {
        start: start,
        isRange: isRange,
        end: end,
        interval: newInterval
      };
    });
  }

  /**
   * A field editor used to add new triggers to a `schedule:TriggerSchedule`.
   * Registered using key `triggers-add`.
   * 
   * @inner
   * @class
   * @name TriggersAddEditor
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */
  var TriggersAddEditor = fe.defineEditor(MobileFieldEditor, /** @lends TriggersAddEditor.prototype */{

    /**
     * Instantiates sub-field editors for start time, range toggle, end
     * time, and interval.
     * 
     * @param {jQuery} element
     */
    doInitialize: function doInitialize(element) {
      var that = this,
          dom = that.jq();

      element.html(triggerAddHtml.patternReplace({
        triggerTime: mobileLex.get('schedule.triggerTime'),
        rangeEnd: scheduleLex.get('trigger.rangeEnd'),
        rangeInterval: scheduleLex.get('trigger.rangeInterval'),
        triggersToCreate: mobileLex.get('schedule.triggersToCreate')
      }));

      //arm handler on range checkbox
      element.on(events.MODIFY_EVENT, '.triggerRange', function () {
        that.rangeEditor.read().then(function (isRange) {
          var startText = mobileLex.get(isRange ? 'schedule.rangeStart' : 'schedule.triggerTime') + ':';

          element.find('.rangeStartLabel').text(startText);

          //show/hide end/interval editors if range is checked
          toggleRangeDisplay(element, isRange);
        }).catch(baja.error);
      });

      element.on(events.MODIFY_EVENT, function () {
        that.updateTriggerCountDisplay();
      });

      return Promise.join(fe.makeFor({
        value: baja.Time.DEFAULT,
        element: $('.triggerStartTime', dom)
      }), fe.makeFor({
        value: false,
        element: $('.triggerRange', dom),
        label: scheduleLex.get('trigger.range') + '?',
        key: 'checkbox'
      }),
      //11:59 PM
      fe.makeFor({
        value: baja.Time.make(baja.RelTime.MILLIS_IN_DAY - 1),
        element: $('.triggerEndTime', dom)
      }), fe.makeFor({
        value: baja.RelTime.make(baja.RelTime.MILLIS_IN_HOUR),
        element: $('.triggerInterval', dom),
        durationOrder: ['h', 'i'] //don't show seconds/days
      })).spread(function (startEd, rangeEd, endEd, intervalEd) {
        that.startEditor = startEd;
        that.rangeEditor = rangeEd;
        that.endEditor = endEd;
        that.intervalEditor = intervalEd;
      });
    },

    /**
     * Updates trigger count display.
     * 
     * @param {baja.Component} value a `schedule:DaySchedule` existing as the
     * `times` slot on a `schedule:TriggerSchedule`.
     */
    doLoad: function doLoad(value) {
      return this.updateTriggerCountDisplay();
    },

    /**
     * Creates an array of `schedule:TimeSchedule`s based on the user input. If
     * the range checkbox is unchecked, this array will be of length 1 and only
     * contain an entry for the start time entered by the user. Otherwise, the
     * start time, end time, and interval will be used to assemble a list of
     * triggers.
     */
    doRead: function doRead() {
      // first save all our field editors...
      return readAddTriggerValues(this).then(function (result) {
        // ...then use the results to assemble an array of time schedules.

        var start = result.start,
            isRange = result.isRange,
            end = result.end,
            interval = result.interval;

        if (isRange) {
          return doRange(start, end, interval);
        } else {
          return [baja.$(TIME_SCHEDULE_TYPE, {
            start: start,
            finish: baja.Time.make(start.getTimeOfDayMillis() + 20000)
          })];
        }
      });
    },

    doSave: function doSave(times) {
      var value = this.value(),
          adds = _.map(times, function (timeSchedule) {
        if (!hasTime(value, timeSchedule.getStart())) {
          return value.add({ slot: 'time?', value: timeSchedule });
        }
      });

      return Promise.all(adds).then(function () {
        return value;
      });
    }
  });

  TriggersAddEditor.prototype.updateTriggerCountDisplay = function () {
    var dom = this.jq();

    return readAddTriggerValues(this).then(function (result) {
      var start = result.start,
          end = result.end,
          interval = result.interval,
          totalMillis = end.getTimeOfDayMillis() - start.getTimeOfDayMillis() + baja.RelTime.MILLIS_IN_MINUTE,
          intervalMillis = interval.getMillis(),
          intervalValid = intervalMillis >= baja.RelTime.MILLIS_IN_MINUTE,
          totalTriggers = Math.max(Math.ceil(totalMillis / intervalMillis), 1),
          label = dom.find('.triggersToCreate'),
          text = intervalValid ? mobileLex.get('schedule.triggersToCreate') + ': ' + totalTriggers : mobileLex.get('schedule.message.intervalRequired');

      label.text(text);
      label.toggleClass('error', !intervalValid);
    });
  };

  fe.register(DAY_SCHEDULE_TYPE, TriggersAddEditor, 'triggers-add');

  return {
    TriggersAddEditor: TriggersAddEditor,
    TriggersListEditor: TriggersListEditor
  };
});
