/**
 * @file A view listing slots of a component for use in a property sheet.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!mobile', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/slot', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/mobile', 'mobile/util/mobile/views/ListView', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/BaseFieldEditor', 'mobile/propsheet/statusgradients', 'bajaux/commands/Command', 'bajaux/events'], function (baja, lexs, $, jqm, Promise, _, slotUtil, dialogs, mobileUtil, ListView, fe, BaseFieldEditor, statusGradients, Command, events) {

  "use strict";

  //local vars

  var encodePageId = mobileUtil.encodePageId,
      mobileLex = lexs[0],
      MODIFY_EVENT = events.MODIFY_EVENT,


  //exports
  PropertySheetListView;

  /**
   * Creates an editable div for a component or a property of a component.
   * This div will be displayed when linking to an editable property of the
   * currently displayed component, like StringWritable.in2, for example.
   * The appropriate field editor will be retrieved and inserted into the
   * created div, which will then be inserted into the target div. 
   * 
   * What you will have afterwards:
   * 
   * 
   *     <div id="element">
   *       <div class="property">
   *         <div class="editor"> <!-- this div created by your field editor -->
   *           <!-- field editor stuff from niagara.fieldEditors here -->
   *         </div>
   *       </div>
   *     </div>
   * 
   * @inner
   * 
   * @param {Object} params an object with extra parameters
   * 
   * @params {baja.Value} params.value the Baja value for which to load a
   * field editor - if omitted, defaults to `params.container.get(params.slot)`
   * 
   * @param {jQuery} params.element the div we are inserting into
   * 
   * @params {baja.Facets} [params.facets] facets that may be passed to the
   * created field editor
   * 
   * @param {baja.Complex} [params.container] the component whose property we are
   * editing.
   * 
   * @param {baja.Slot} [params.slot] the slot the edited object lives in
   * 
   * @see niagara.fieldEditors.makeFor
   */
  function loadEditor(params) {

    baja.strictArg(params, Object);

    params.facets = params.facets || {};

    return fe.makeFor(params);
  }

  function defer() {
    var res,
        rej,
        promise = new Promise(function (a, b) {
      res = a;rej = b;
    });
    return { resolve: res, reject: rej, promise: promise };
  }

  function getComponentName(component) {
    if (!(component instanceof baja.Complex)) {
      return undefined;
    }

    return component.getDisplayName() || component.getName();
  }

  /**
   * Forms the content listview of a property sheet.
   * 
   * @class
   * @extends niagara.util.mobile.ListView
   * @memberOf niagara.propSheet
   */
  PropertySheetListView = baja.subclass(function PropertySheetListView() {
    var that = this,
        saveCommand = new Command({
      displayName: "%lexicon(mobile:save)%",
      enabled: false,
      func: function func() {
        return that.save();
      }
    }),
        refreshCommand = new Command({
      displayName: "%lexicon(mobile:refresh)%",
      func: function func() {
        return that.refresh();
      }
    });

    baja.callSuper(PropertySheetListView, that, arguments);
    that.fieldEditorMap = {};

    /**
     * Command to save this property sheet (invisible until sheet is set
     * modified)
     * @field
     * @private
     * @type {Command}
     */
    that.$saveCommand = saveCommand;

    /**
     * Command to refresh this property sheet
     * @field
     * @private
     * @type {Command}
     */
    that.$refreshCommand = refreshCommand;
    that.getCommandGroup().add(saveCommand, refreshCommand);
  }, ListView);

  /**
   * After initializing the DOM element, will listen for field editor
   * modified events, and mark myself modified if one of my field editors
   * is modified.
   * 
   * @name niagara.propSheet.PropertySheetListView#doInitialize
   * @function
   */
  PropertySheetListView.prototype.doInitialize = function (element) {
    var that = this;

    return ListView.prototype.doInitialize.call(this, element).then(function () {
      element.on(MODIFY_EVENT, function (event, view) {
        if (view instanceof BaseFieldEditor) {
          that.setModified(true);
        }
      });
    });
  };

  PropertySheetListView.prototype.doLoad = function () {
    var that = this;
    return ListView.prototype.doLoad.apply(this, arguments).then(function () {
      var batch = new baja.comm.Batch(),
          prom = that.doSubscribing(batch);
      batch.commit();
      return prom;
    });
  };

  PropertySheetListView.prototype.doDestroy = function () {
    var that = this;
    return Promise.resolve(ListView.prototype.doDestroy.apply(that, arguments)).then(function () {
      var batch = new baja.comm.Batch(),
          prom = that.doUnsubscribing(batch);
      batch.commit();
      return prom;
    });
  };

  /**
   * When property sheet is set to modified, set the save Command to visible,
   * and vice versa.
   * 
   * @name niagara.propSheet.PropertySheetListView#setModified
   * @function
   * @param {Boolean} modified
   */
  PropertySheetListView.prototype.setModified = function (modified) {
    baja.callSuper('setModified', PropertySheetListView, this, arguments);

    this.$saveCommand.setEnabled(modified);
  };

  PropertySheetListView.prototype.toDisplayName = function () {
    var complex = this.value(),
        name = getComponentName(complex);

    return Promise.resolve(name || mobileLex.get('propsheet.station'));
  };

  /**
   * Loads the field editor divs for the property about to be edited.
   * 
   * @name niagara.propSheet.PropertySheetListView#loadEditorParams
   * @function
   * @private
   * 
   * @param {Object} params an object literal
   * @param {baja.Complex} params.container the object containing the property 
   * we are to be editing - for example, if we are editing 
   * `myNumericWritable.in2`, then the container is `myNumericWritable`
   * @param {baja.Slot} params.slot the Slot corresponding to the property we 
   * will edit
   * @param {jQuery} params.element the div into which the field editor div 
   * will be inserted
   * @returns {Promise} will be resolved with the editor instance when the 
   * editor has been fully loaded
   */
  PropertySheetListView.prototype.loadEditorParams = function (params) {
    baja.strictArg(params, Object);
    baja.strictArg(params.container, baja.Complex);
    baja.strictArg(params.slot, baja.Slot);
    baja.strictArg(params.element, $);

    var that = this,
        spinner = mobileUtil.spinnerTicket(1000);

    return loadEditor(params).then(function (editor) {
      that.fieldEditorMap[params.slot] = editor;
    }).finally(function () {
      spinner.hide();
    });
  };

  /**
   * Override point to show field editors when expanding an editable property.
   * 
   * @name niagara.propSheet.PropertySheetListView#populateExpandedDiv
   * @function
   * 
   * @param {baja.Property} prop the property being expanded
   * @param {jQuery} expandedDiv the div that is expanded to show property
   * details
   */
  PropertySheetListView.prototype.populateExpandedDiv = function (prop, expandedDiv) {
    var ul = this.jq().children('ul'),
        pageId = encodePageId(prop);

    return this.loadEditorParams({
      container: this.value(),
      slot: prop,
      element: expandedDiv
    }).then(function () {
      expandedDiv.on(MODIFY_EVENT, function (event, editor) {
        ul.children('li#' + pageId).find('div.display').addClass('dirty');
      });
    }, baja.error);
  };

  /**
   * Saves all outstanding changes (dirty field editors) on the property
   * sheet and commits the changes up to the station.
   * 
   * Validates and then saves a collection of modified editors. If any of the
   * validation steps fail, none of the save steps will be performed. If an
   * editor has not been modified (isModified() === false) then it will not
   * be saved.
   * 
   * @name niagara.propSheet.PropertySheetListView#save
   * @function
   * @returns {Promise}
   */
  PropertySheetListView.prototype.save = function () {
    var that = this,
        batch = new baja.comm.Batch(),
        spinner = mobileUtil.spinnerTicket(1000);

    var dirtyEds = _.filter(that.fieldEditorMap, function (ed) {
      return ed.isModified();
    });

    var validates = _.map(dirtyEds, function (ed) {
      return ed.validate();
    });

    return Promise.all(validates).then(function () {
      var saves = _.map(dirtyEds, function (editor) {
        return editor.save({ batch: batch });
      });

      setTimeout(function () {
        batch.commit();
      }, 150); //TODO: need commitReady event here too.

      return Promise.all(saves);
    }).then(function () {
      that.jq().find('div.display.dirty').removeClass('dirty').addClass('editable');
      that.setModified(false);
    }).finally(spinner.hide());
  };

  PropertySheetListView.prototype.refresh = function () {
    var that = this;

    if (!that.isModified()) {
      return that.load(that.value());
    }

    return that.toDisplayName().then(function (displayName) {
      var df = defer();
      dialogs.confirmAbandonChanges({
        yes: function yes(cb) {
          that.save().then(function () {
            return that.load(that.value());
          }).then(function () {
            df.resolve();
            cb.ok();
          }, function (err) {
            df.reject(err);
            cb.fail(err);
          });
        },
        no: function no(cb) {
          that.setModified(false);
          that.load(that.value()).then(function () {
            df.resolve();
            cb.ok();
          }, function (err) {
            df.reject(err);
            cb.fail(err);
          });
        },
        viewName: displayName
      });

      return df.promise;
    });
  };

  /**
   * Performs subscription on this component's child components. The default 
   * behavior when a child component changes is simply to update the display
   * value on this navigator with the child component's new display value.
   * 
   * @name niagara.propSheet.PropertySheetListView#doSubscribing
   * @function
   * @private
   * 
   * @param {baja.comm.Batch} batch a batch to use to subscribe children (should
   * be the same batch used to subscribe the parent component)
   */
  PropertySheetListView.prototype.doSubscribing = function (batch) {
    var childrenSub = this.childrenSubscriber,
        that = this,
        value = that.value(),
        subscribableChildren = [];

    if (!value.getType().isComponent()) {
      return;
    }

    if (!childrenSub) {
      childrenSub = new baja.Subscriber();

      childrenSub.attach('changed', function (subprop, cx) {
        that.updateDisplay(this.getPropertyInParent());
      });

      this.childrenSubscriber = childrenSub;
    }

    value.getSlots(function (slot) {
      if (!slot.isProperty()) {
        return false;
      }
      var type = slot.getType();
      return type.isComponent() && (type.is("baja:IStatus") || type.is("baja:VirtualComponent"));
    }).each(function (slot) {
      var comp = this.get(slot);
      if (!comp.isSubscribed()) {
        subscribableChildren.push(this.get(slot));
      }
    });

    return childrenSub.subscribe({
      comps: subscribableChildren,
      batch: batch
    });
  };

  /**
   * Unsubscribes from all this component's child components.
   * 
   * @name niagara.propSheet.PropertySheetListView#doUnsubscribing
   * @function
   * @private
   * 
   * @param {baja.comm.Batch} batch a batch to use to unsubscribe children 
   * (should be the same batch used to unsubscribe the parent component)
   */
  PropertySheetListView.prototype.doUnsubscribing = function (batch) {
    var childSub = this.childrenSubscriber,
        value = this.value();

    if (!childSub || !value.getType().isComponent()) {
      return;
    }

    return childSub.unsubscribeAll({ batch: batch });
  };

  function setStatus(element, status) {
    statusGradients.applyStatusCSS(element, status);
  }

  /**
   * A PropertySheet when making a list item will add a CSS class `editable` to
   * the value display div if the property can be edited.
   * 
   * @name niagara.propSheet.PropertySheetListView#makeListItem
   * @function
   */
  PropertySheetListView.prototype.makeListItem = function (complex, prop) {
    return baja.callSuper('makeListItem', PropertySheetListView, this, arguments).then(function (li) {
      var status,
          statusBg = $('<div class="statusBg"></div>').prependTo(li);

      if (slotUtil.isEditable(prop)) {
        li.find('div.display').addClass('editable');
      }

      if (prop.getType().is('baja:IStatus')) {
        status = baja.Status.getStatusFromIStatus(complex.get(prop));
        setStatus(statusBg, status);
      }
      return li;
    });
  };

  /**
   * Called to update the display value of a PropertySheet item.
   * 
   * @name niagara.propSheet.PropertySheetListView#updateDisplay
   * @function
   */
  PropertySheetListView.prototype.updateDisplay = function (prop) {
    baja.callSuper('updateDisplay', PropertySheetListView, this, arguments);

    var li, status;

    if (prop.getType().is('baja:IStatus')) {
      li = this.jq().find('li.property#' + encodePageId(prop));
      status = baja.Status.getStatusFromIStatus(this.value().get(prop));
      setStatus(li.find('div.statusBg'), status);
    }
  };

  PropertySheetListView.prototype.makeListItemParams = function (complex, prop) {
    return baja.callSuper('makeListItemParams', PropertySheetListView, this, arguments).then(function (params) {
      if (params.expandable) {
        return fe.isRegistered(prop).then(function (isRegistered) {
          if (!isRegistered) {
            params.expandable = false;
          }
          return params;
        });
      }

      return params;
    });
  };

  return PropertySheetListView;
});
