/**
 * @file Functions relating to the BaseFieldEditor object that all
 * field editors in Niagara mobile apps extend from. Also, functions relating
 * to registering field editors on Types.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

/**
 * @private
 * @module mobile/fieldeditors/fieldeditors
 */
define(['baja!mobile:MobileFieldEditor', 'baja!', 'jquery', 'Promise', 'underscore', 'mobile/util/aop', 'mobile/util/slot', 'nmodule/js/rc/asyncUtils/asyncUtils'], function (types, baja, $, Promise, _, aop, slotUtil, asyncUtils) {

  'use strict';

  var //private vars
  fieldEditorMap = {},
      defaultFacets = baja.Facets.make({
    timeFormat: baja.getTimeFormatPattern(),
    unitConversion: baja.getUnitConversion()
  }),


  //imports
  doRequire = asyncUtils.doRequire,


  //constants
  MOBILE_FIELD_EDITOR_TYPE = 'mobile:MobileFieldEditor',
      WORKBENCH_FIELD_EDITOR_TYPE = 'workbench:WbFieldEditor',
      FIELD_EDITOR_FACET = 'fieldEditor',


  //exports
  composite = {};

  ////////////////////////////////////////////////////////////////
  //Utility Functions
  ////////////////////////////////////////////////////////////////

  /**
   * Wraps a field editor in a new div with a label attached.
   * 
   * @memberOf niagara.fieldEditors
   */
  function toLabeledEditorContainer(labelText, targetElement) {
    var containerDiv = $('<div class="labeledEditor"/>').appendTo(targetElement),
        label = $('<label class="editorLabel" />'),
        editorContainer = $('<div class="editorContainer"/>');

    if (labelText !== undefined) {
      label.text(labelText);
    }

    containerDiv.append(label).append(editorContainer);
    return editorContainer;
  }

  ////////////////////////////////////////////////////////////////
  //Field Editor Registry
  ////////////////////////////////////////////////////////////////


  /**
   * @memberOf niagara.fieldEditors
   * @param component
   */
  function toSaveDataComponent(component) {
    var copy = component.newCopy(true);
    copy.editedSlots = {};

    copy.getSlots().properties().isComplex().each(function (slot) {
      var subComponent = copy.get(slot);
      copy.set({
        slot: slot,
        value: toSaveDataComponent(subComponent)
      });
    });

    aop.after(copy, 'set', function (args, value) {
      var obj = args[0];
      this.editedSlots[String(obj.slot)] = obj.value;
      return value;
    });

    return copy;
  }

  /**
   * The main method to define a field editor. This method takes in three
   * methods implementing the three primary behaviors of a field editor.
   * 
   * @memberOf niagara.fieldEditors
   * @see module:bajaux/Widget
   * 
   * @param {Function} FieldEditor the field editor constructor we wish to
   * subclass, e.g. `niagara.fieldEditors.BaseFieldEditor` or
   * `niagara.fieldEditors.mobile.MobileFieldEditor`
   * 
   * @param {Object} params an object holding parameters
   * @param {Function} params.doInitialize a function that will create the 
   * necessary HTML (returned as a jQuery div object), including such elements 
   * as text inputs, select menus, and checkboxes, that define the behavior of 
   * this field editor.
   * 
   * @param {Function} params.doLoad a method that will populate the HTML 
   * elements of the div created in the previous step with the given value. For 
   * instance, if this were a String editor, this function would take in a 
   * baja:String and simply set the value of its text input element. 
   * 
   * @param {Function} params.doRead a function that will retrieve the 
   * current values of the input elements of the editor div and assemble them 
   * into an object. The output of this function will be passed into both
   * `validate()` (if provided) and `doSave`.
   * 
   * @param {Function} [params.doSave] a function that will commit any changes
   * made to this field editor. If editing a `baja:Simple`, it will simply build
   * a new instance of the edited type using user-entered data (edit-by-value
   * semantics). If editing a `baja:Complex`,  then user-entered data in the
   * field editor will be applied directly to the object itself
   * (edit-by-reference semantics). If editing a  subscribed/mounted component,
   * an asynchronous call to the server must be made to commit those changes as
   * well.
   * 
   * This function can be omitted entirely from your field editor definition.
   * If omitted, `defaultDoSave` will be used, which should be fine for most
   * purposes. Please see the documentation for that function for information
   * about the expected output from `doRead()`.
   * 
   * @param {Function} [params.postCreate] an optional function that will 
   * execute after the field editor's constructor completes. This function may
   * perform more specialized initialization on the field editor.
   * 
   * @param {Function} [params.validate] an optional function that performs
   * validation logic on this field editor. This function will _not_ be
   * called automatically - if you wish to perform validation on a field editor
   * then `validate()` must be called before `save`.
   * 
   * @see module:mobile/fieldeditors/BaseFieldEditor
   * @see niagara.fieldEditors.defaultDoSave
   * 
   * @returns {Function} a constructor function for this field editor type.
   * The function returned from `defineEditor` must be instantiated  using "new"
   * to obtain an actual field editor object.
   */
  function defineEditor(FieldEditor, params) {
    params = baja.objectify(params);

    var doInitialize = params.doInitialize,
        doLoad = params.doLoad,
        doRead = params.doRead,
        doSave = params.doSave,
        ctor;

    baja.strictAllArgs([doInitialize, doLoad], [Function, Function]);

    ctor = baja.subclass(function () {
      baja.callSuper(ctor, this, arguments);
    }, FieldEditor);

    ctor.prototype.doInitialize = function (element) {
      var that = this;

      return Promise.resolve(FieldEditor.prototype.doInitialize.call(that, element)).then(function () {
        return doInitialize.call(that, element);
      });
    };

    ctor.prototype.doLoad = function (value) {
      var that = this;

      return Promise.resolve(FieldEditor.prototype.doLoad.call(that, value)).then(function () {
        return doLoad.call(that, value);
      });
    };

    if (typeof doRead === 'function') {
      ctor.prototype.doRead = doRead;
    }

    if (typeof doSave === 'function') {
      ctor.prototype.doSave = doSave;
    }

    if (params.validate) {
      aop.after(ctor.prototype, 'postCreate', function (args) {
        this.validators().add(params.validate);
      });
    }

    if (params.postCreate) {
      aop.after(ctor.prototype, 'postCreate', function (args) {
        params.postCreate.call(this);
      });
    }

    if (params.doEnabled) {
      aop.after(ctor.prototype, 'doEnabled', function (args) {
        params.doEnabled.apply(this, args);
      });
    }

    return ctor;
  }

  /**
   * Returns true if the given type is either a mobile or workbench field
   * editor type (extends `mobile:MobileFieldEditor` or 
   * `workbench:WbFieldEditor`).
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {String|Type} type
   * @returns {Promise} promise to be resolved with true or false
   */
  function isFieldEditorType(type) {
    if (type) {
      return baja.importTypes({ typeSpecs: [type] }).then(function (types) {
        var type = types[0];
        if (type.is(MOBILE_FIELD_EDITOR_TYPE)) {
          return true;
        } else if (type.is(WORKBENCH_FIELD_EDITOR_TYPE)) {
          return true;
        } else {
          baja.error("fieldEditor facet " + type + " does not reference a " + "sub-Type of MobileFieldEditor.");
          return false;
        }
      }, function (err) {
        //does have a fieldEditor facet, but the Type is unknown.
        //possible fat-finger
        baja.error("fieldEditor facet " + type + " does not reference a " + "known Type.");
        return false;
      });
    } else {
      return Promise.resolve(false);
    }
  }

  /**
   * Checks to see if a field editor/retriever is registered for this particular
   * Baja value.
   * 
   * @memberOf niagara.fieldEditors
   * @param {baja.Value|Type} value the value to check for a registered
   * editor. If this is a `Type`, will only check for editors that have been 
   * explicitly registered through `niagara.fieldEditors.register`  or that have
   * a `mobile:MobileFieldEditor` declared as an agent  on it. If this is a
   * `baja.Value`, will first check using the value's `Type`, and failing that,
   * will check for a `fieldEditor` facet declared on it.
   * @returns {Promise} promise to be resolved with true if this type has been
   * registered
   */
  function isRegistered(value) {
    function getType() {
      if (typeof value === 'string') {
        return baja.importTypes({ typeSpecs: [value] }).then(function (types) {
          return types[0];
        });
      } else if (value.getType) {
        return Promise.resolve(value.getType());
      } else {
        return Promise.resolve(value);
      }
    }

    function hasRegisteredEditor(type) {
      var cfe = getCustomFieldEditors();

      while (type) {
        //i've explicitly registered an editor using fieldEditors.register
        if (fieldEditorMap[String(type)]) {
          return true;
        }
        //i have a MobileFieldEditor declared as an agent on this type
        if (cfe[String(type)]) {
          return true;
        }
        type = type.getSuperType();
      }
    }

    if (value !== null && value !== undefined) {
      return getType().then(function (type) {
        if (hasRegisteredEditor(type)) {
          return true;
        } else {
          //no field editors registered - let's see if the value has a
          //fieldEditor facet on it
          var facets = value.getFacets && value.getFacets(),
              feFacet = facets && facets.get(FIELD_EDITOR_FACET);

          return isFieldEditorType(feFacet);
        }
      });
    } else {
      return Promise.resolve(false);
    }
  }

  function isRegisteredFieldEditor(value) {
    return isFieldEditorType(value).then(function (result) {
      if (!result) {
        return false;
      } else {
        return isRegistered(value);
      }
    });
  }

  /**
   * Creates a URL to the given JS file ORD. This should only be a path to
   * a MobileFieldEditor JS file, not just any arbitrary file.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {String} js the direct path to the JS file
   * @param {String|Type} type the type of the  `mobile:MobileFieldEditor`
   * whose JS resources we're loading
   * @param {Array} agentOn an array of the type specs of all the types this
   * `mobile:MobileFieldEditor` is registered as an agent on.
   * @see niagara.fieldEditors.doPreloadFromSpec
   */
  function createJsPath(js, type, agentOn) {
    var path = '/ord/' + js;
    path += '|view:mobile:MobileFieldEditorView%3FfeTypeSpec=' + type + ';typeSpecs=' + agentOn.join(',');
    return path;
  }

  /**
   * Performs the preload of all JS/CSS resources declared by the given
   * MobileFieldEditor type. These resources are declared in
   * `niagara.view.fieldEditorResources` which is itself populated when the page
   * is generated in  `com.tridium.mobile.BDefaultMobileWebProfile`.
   * 
   * The requests for the JS files will be routed through the
   * `mobile:MobileFieldEditorView`.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type|String} type the `mobile:MobileFieldEditor` type whose
   * JS/CSS resources we need to load
   */
  function doPreloadFromSpec(type) {
    type = String(type);

    var fer = getFieldEditorResources(),
        cfe = getCustomFieldEditors(),
        resources = fer && fer[type],
        toRequire = [];

    //type is not a MobileFieldEditor, or it's been loaded already
    if (!resources) {
      return Promise.resolve();
    }

    _.each(resources.js, function (js) {
      toRequire.push(createJsPath(js, type, resources.agentOn));
    });

    _.each(resources.css, function (css) {
      if (css.indexOf('.css') === css.length - 4) {
        css = css.substring(0, css.length - 4);
      }
      toRequire.push('css!' + css);
    });

    return doRequire(toRequire).then(function () {
      //all loaded, we'll never have to load these again
      delete fer[type];
      delete cfe[type];
    });
  }

  /**
   * Any particular type may have a MobileFieldEditor declared as an agent on
   * it. If this is the case, we should load the JS for that MobileFieldEditor
   * and use that JS to instantiate the field editor for that type. This
   * function will look at all the MobileFieldEditors declared as agents on the
   * given type, and perform the preloads (in parallel) for those 
   * MobileFieldEditors.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type|String} type the type of the object we're trying to 
   * instantiate a field editor for
   * @see niagara.fieldEditors.doPreloadFromSpec
   */
  function doPreloadFromAgents(type) {
    type = String(type);

    var cfe = getCustomFieldEditors(),
        customFETypes = cfe && cfe[type];

    //no agents declared on this type
    if (!customFETypes) {
      return Promise.resolve();
    }

    //preload all the field editors declared as agents on this type
    var preloads = _.map(customFETypes, function (feType) {
      return doPreloadFromSpec(feType);
    });

    return Promise.all(preloads).then(function () {
      //all agents loaded, no need to ever check this type again
      delete cfe[type];
    });
  }

  /**
   * Performs the actual checking and loading of custom field editor code on the
   * given Type.
   * 
   * If the type has an entry in `niagara.view.customFieldEditors`, then for
   * each corresponding `BMobileFieldEditor` type spec in
   * `niagara.view.fieldEditorResources`, it will dynamically load all declared
   * Javascript code.
   * 
   * Once all the JS resources have been downloaded and executed, the entries
   * will be deleted from `niagara.view.customFieldEditors` and
   * `niagara.view.fieldEditorResources`, to prevent having to check them a
   * second time, and the promise will be resolved.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type} type the type to check for custom field editor code
   * @returns {Promise}
   */
  function doPreloadCustomFieldEditor(type) {

    if (type.is(MOBILE_FIELD_EDITOR_TYPE)) {
      return doPreloadFromSpec(type);
    } else {
      return doPreloadFromAgents(type);
    }
  }

  /**
   * Checks for any custom field editor code registered on the given Type via
   * `BMobileFieldEditor` instances. It does this by checking
   * `niagara.view.customFieldEditors`, which is built up server-side in
   * `BDefaultMobileWebProfile`.
   * 
   * Recursively checks the given type and all its super types, passing each
   * one in turn to `doPreloadCustomFieldEditor`.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * 
   * @param {Type} type the type for check for custom field editor code
   * @see niagara.fieldEditors.doPreloadCustomFieldEditor
   * @returns {Promise}
   */
  function preloadCustomFieldEditor(type) {
    if (type) {
      return doPreloadCustomFieldEditor(type).then(function () {
        return preloadCustomFieldEditor(type.getSuperType());
      });
    } else {
      return Promise.resolve();
    }
  }

  /**
   * Given a type, and optional key, returns the field editor constructor 
   * registered for that type. The key allows multiple different field editors
   * to be registered on a particular Type - e.g., the RelTime editor used when
   * selecting an override duration has a select dropdown appended and is
   * registered using the key 'override'.
   * 
   * If there is no field editor registered for the type as given, this
   * method walks up the supertype chain until one is found. If none is found,
   * it defaults to a string-based field editor.
   * 
   * @private
   * @memberOf niagara.fieldEditors
   * 
   * @param {Type} type The type for which to find an editor/retriever
   * @param {String} [key] The String key under which the editor is registered.
   * If no key is given, `'default'` is used.
   * @returns {Promise.<Function>} a field editor constructor registered for
   * this type
   */
  function doGet(type, key) {
    var got;

    if (!type) {
      return Promise.reject("cannot get field editor for null type");
    }

    if (typeof type === 'string') {
      type = baja.lt(type);
    }

    key = key || 'default';

    return preloadCustomFieldEditor(type).then(function () {
      while (!got && type) {
        got = fieldEditorMap[String(type)];
        if (got && got[key]) {
          return got[key];
        } else {
          type = type.getSuperType();
        }
      }

      return doGet('baja:String', 'default');
    });
  }

  /**
   * Register a field editor and retriever for a given type.
   * 
   * @memberOf niagara.fieldEditors
   * @param {Type|String} type The type to register
   * @param {Function} editor The field editor constructor for this type
   * @param {String} [key] An optional key for registering a special type of
   * editor for this Type. If no key is given, `'default'` is used.
   */
  function register(type, editor, key) {
    if (!type) {
      throw new Error("cannot register field editor for null type");
    }

    if (typeof editor !== 'function') {
      throw new Error("editor is required");
    }

    type = String(type);
    key = key || 'default';

    var obj = fieldEditorMap[type] || {};
    obj[key] = editor;
    fieldEditorMap[type] = obj;
    return editor;
  }

  function computeFacets(editedObject, container, slot, inpFacets) {
    var facets = defaultFacets,
        myFacets;

    if (container && slot) {
      myFacets = slotUtil.getFacets(container, slot);
    } else {
      myFacets = slotUtil.getFacets(editedObject);
    }

    if (myFacets) {
      facets = baja.Facets.make(facets, myFacets);
    }
    if (inpFacets) {
      facets = baja.Facets.make(facets, inpFacets);
    }

    return facets;
  }

  function getNiagaraView() {
    return typeof niagara !== 'undefined' && niagara.view || {};
  }

  function getCustomFieldEditors() {
    return getNiagaraView().customFieldEditors || {};
  }

  function getFieldEditorResources() {
    return getNiagaraView().fieldEditorResources || {};
  }

  /**
   * Instantiates a field editor for a Baja value.
   * 
   * The `params` input object must have at least one of the following: a
   * `value` property, or `container` and `slot` properties. If `value` is
   * present, a field editor will always be constructed for that value. If 
   * `value` is omitted, `container` and `slot` _must_ be present - `value`
   * will then default to `container.get(slot)`.
   * 
   * If a container and slot are specified as properties of the `params`
   * object, then calling `save` on this  field editor will cause the edited
   * object to be committed/saved onto the  container. This is most useful when
   * editing a `BSimple`  property of a `BComplex`. If we omitted container and
   * slot, we would have to perform `component.set({...})` ourselves - by 
   * including container and slot, the field editor will do the work for us.
   * 
   * @memberOf niagara.fieldEditors
   * 
   * @param {Object} params
   * 
   * @param {baja.Value} params.value the object we are editing. If omitted,
   * will default to `params.container.get(params.slot)`.
   * 
   * @param {baja.Component} [params.container] the component containing the 
   * property we are editing. If omitted, and if `params.value` is a  mounted
   * component, will default to `editedObject.getParent()`.
   * 
   * @param {baja.Slot} [params.slot] the slot defining where the edited
   * object  lives - only used if a container is used as well. If omitted, and
   * if `params.value` is a mounted component, will default to
   * `params.value.getPropertyInParent()`.
    * @param {String} [params.key] a key defining a special type of field
   * editor we want to retrieve for this object - must have already been
   * registered using `niagara.fieldEditors.register()`
   * 
   * @param {module:mobile/fieldeditors/BaseFieldEditor} [params.parent] defines
   * this field editor as being a child editor of a parent editor. In this case,
   * calling `setModified()` on this editor will cause the parent editor to be
   * marked as modified as well.
   * 
   * @param {String|Type} [params.type] specifies the type spec we want a
   * field editor for. Only to be used in very special circumstances where we
   * are editing an object of one type, using a field editor ordinarily used for
   * another type. If omitted (and in most cases should be), will default to
   * `editedObject.getType()`. 
   * 
   * @param {baja.Facets|Object} [params.facets] a Facets object that can be
   * applied to the created field editor. Input Facets always take priority over
   * facets that are retrieved inside the field editor from  `this.slot` etc,
   * and can be used in any of your field editor's functions by referencing
   * `this.facets`. This can also be an object literal.
   * 
   * @param {Boolean} [params.readonly] set to true if the field editor should
   * be created in readonly mode.
   * 
   * @returns {Promise} will be resolved with an instance of a field 
   * editor for this object
   */
  function makeFor(Ctor, params) {
    if (arguments.length < 2) {
      params = Ctor;
      Ctor = null;
    }

    try {
      baja.strictArg(params);
      if (params.autoInitialize !== false) {
        baja.strictArg(params.element);
      }
    } catch (e1) {
      return Promise.reject("element is required to initialize - did you " + "mean to set autoInitialize = false?");
    }

    if (params.value === undefined) {
      try {
        baja.strictArg(params.container, baja.Complex);
      } catch (e2) {
        return Promise.reject("container is required when value is undefined");
      }
      params.value = params.container.get(params.slot);
    }

    var editedObject = params.value,
        autoInitialize = params.autoInitialize !== false,
        element = params.element,
        type = params.type || editedObject.getType(),
        feType,
        container,
        slot,
        key = params.key;

    //TODO: throw expected errors for undefined
    container = params.container || editedObject.getParent && editedObject.getParent();
    slot = params.slot || editedObject.getPropertyInParent && editedObject.getPropertyInParent();

    params.facets = computeFacets(editedObject, container, slot, params.facets);

    //check to see if the component has a fieldEditor facet expressly registered
    feType = params.facets.get(FIELD_EDITOR_FACET);

    return isRegisteredFieldEditor(feType).then(function (result) {
      if (result) {
        //hey, we do. load this kind of field editor instead. it will be
        //registered as an editor on the actual MobileFieldEditor subtype,
        //so use this type to retrieve the field editor instance but still load 
        //the old value into it
        type = baja.lt(feType);
      }

      function buildAndLoad(FieldEditor) {
        var editor = new FieldEditor(editedObject, container, slot, params);
        if (autoInitialize) {
          return editor.initialize(element).then(function () {
            return editor.load(editedObject);
          }).then(function () {
            return editor;
          });
        } else {
          return editor;
        }
      }

      if (Ctor) {
        return buildAndLoad(Ctor);
      } else {
        return doGet(type, key).then(buildAndLoad);
      }
    });
  }

  /**
   * @namespace
   * @name niagara.fieldEditors
   */
  return {
    composite: composite,

    defineEditor: defineEditor,
    isRegistered: isRegistered,
    makeFor: makeFor,
    register: register,
    toLabeledEditorContainer: toLabeledEditorContainer,
    toSaveDataComponent: toSaveDataComponent
  };
});
