/**
 * @file Field editors for use in a JQM app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/fieldeditors/MobileFieldEditor
 */
define(['baja!', 'jquery', 'jquerymobile', 'mobile/fieldeditors/BaseFieldEditor'], function (baja, $, jqm, BaseFieldEditor) {

  "use strict";

  /**
   * A field editor subclass that makes use of JQM widgets for Niagara Mobile
   * apps.
   * 
   * @class
   * @alias module:mobile/fieldeditors/MobileFieldEditor
   * @extends module:mobile/fieldeditors/BaseFieldEditor
   */

  var MobileFieldEditor = baja.subclass(function MobileFieldEditor(value, container, slot, params) {
    BaseFieldEditor.call(this, value, container, slot, params);
  }, BaseFieldEditor);

  /**
   * After setting the field editor value, this method will be called to refresh
   * any widgets to reflect the updated value. On a mobile field editor, 
   * `refresh` methods will be called on any JQM widgets container in the field
   * editor's DOM.
   */
  MobileFieldEditor.prototype.refreshWidgets = function refreshWidgets() {
    var dom = this.jq(),
        select = dom.find('select'),
        input = dom.find('input'),
        switches = select.filter('.ui-slider-switch'),
        dropdowns = select.filter('.ui-select select'),
        checkboxes = input.filter('.ui-checkbox input'),
        dateboxes = input.filter('.ui-input-datebox input');

    if (switches.length) {
      switches.slider('refresh');
    }
    if (dropdowns.length) {
      dropdowns.selectmenu('refresh');
    }
    if (checkboxes.length) {
      checkboxes.checkboxradio('refresh');
    }
    if (dateboxes.length) {
      dateboxes.datebox('refresh');
    }

    dom.removeClass('ui-br');
  };

  /**
   * Sets a mobile field editor's enabled status by calling `disable` or
   * `enable` methods on any editable JQM widgets contained in the field
   * editor's DOM.
   * 
   * @param {Boolean} enabled
   */
  MobileFieldEditor.prototype.doEnabled = function doEnabled(enabled) {
    var that = this,
        dom = this.jq();

    if (dom) {
      //i.e. initialize() has completed
      that.updateReadonlyStatus(dom);
    }
  };

  MobileFieldEditor.prototype.updateReadonlyStatus = function updateReadonlyStatus(element) {
    var enabled = this.isEnabled(),
        select = element.find('select'),
        input = element.find('input'),
        action = enabled ? 'enable' : 'disable',
        textareas = element.find('textarea'),
        sliders = select.filter(':jqmData(role=slider)'),
        selects = select.filter(':jqmData(role!=slider)'),
        checkboxes = input.filter('[type="checkbox"]'),
        dateboxes = input.filter(':jqmData(role="datebox")'),
        textinputs = input.filter('[type="text"]');

    if (sliders.length) {
      sliders.slider(action);
    }

    if (selects.length) {
      selects.selectmenu(action);
    }

    if (checkboxes.length) {
      checkboxes.checkboxradio(action);
    }

    if (dateboxes.length) {
      dateboxes.datebox(action);
    }

    if (textinputs.length) {
      textinputs.textinput(action);
    }

    if (textareas.length) {
      textareas.textinput(action);
    }
  };

  /**
   * Initializes the HTML created in `makeFor`, triggering the creation of any
   * JQM widgets, setting the value, and arming change handlers.
   * 
   * @param {jQuery} element
   * @returns {Promise}
   */
  MobileFieldEditor.prototype.initialize = function (element) {
    var that = this;

    return BaseFieldEditor.prototype.initialize.call(this, element).then(function () {
      element.enhanceWithin();
      that.updateReadonlyStatus(element);
    });
  };

  /**
   * Refreshes any JQM widgets when a new value is loaded.
   *  
   * @param {baja.Value} value
   * @returns {Promise}
   */
  MobileFieldEditor.prototype.load = function (value) {
    var that = this;

    return BaseFieldEditor.prototype.load.call(that, value).then(function () {
      that.refreshWidgets();
    });
  };

  return MobileFieldEditor;
});
