/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define([], function () {

  'use strict';
  
  var REG = '$$events';

  function getHandlers(obj, name) {
    if (!name) {
      throw 'event name required';
    }
    var e = obj[REG] || (obj[REG] = {});
    return e[name] || (e[name] = []);
  }
  
  function getAllEventNames(obj) {
    var names = [],
        events = obj[REG];
    if (events) {
      for (var name in events) {
        if (events.hasOwnProperty(name)) {
          names.push(name);
        }
      }
    } 
    return names;
  }
  
  function isEmpty(obj) {
    if (obj) {
      for (var name in obj) {
        if (obj.hasOwnProperty(name)) {
          return false;
        }
      }
    }
    return true;
  }
  
  function clear(arr) { arr.splice(0, arr.length); }
  
  function cleanup(obj) {
    if (isEmpty(obj[REG])) { delete obj[REG]; }
  }

  /**
   * @mixin tinyevents
   */
  var fns = {
    /**
     * Emit an event.
     * @param name
     * @returns {Array.<*>} the results of all the handler calls, or an empty
     * array if there are no handlers defined.
     */
    emit: function (name) {
      var that = this,
          handlers = getHandlers(that, name);
      if (handlers) {
        var args = Array.prototype.slice.call(arguments, 1);
        //TODO: should this return Promise.all?
        return handlers.map(function (handler) {
          return handler.apply(that, args);
        });
      } else {
        return [];
      }
    },

    on: function (name, func) {
      if (!func) {
        throw 'handler required';
      }
      var handlers = getHandlers(this, name);
      if (handlers.indexOf(func) === -1) {
        handlers.push(func);
      }
      return this;
    },

    //TODO: how about an afterTriggered method
    //for example, i want to do something but only after an editor is initialized
    //editor does trigger an initialized event, but only once
    //therefore on('initialized') won't work if initialization is already complete
    //hold onto triggered status for certain events and trigger immediately from on()
    //after implementing this, revisit all calls to .on('initialized'), .on('loaded')

    once: function (name, func) {
      return this.on(name, function f() {
        func.apply(this, arguments);
        this.removeListener(name, f);
      });
    },

    removeListener: function (name, func) {
      var that = this,
          handlers = getHandlers(that, name),
          i = handlers.indexOf(func);

      if (i !== -1) {
        handlers.splice(i, 1);
      }
      
      if (!handlers.length) {
        delete that[REG][name];
      }
      
      cleanup(that);
      
      return this;
    },

    removeAllListeners: function (name) {
      var that = this;

      if (name) {
        clear(getHandlers(that, name));
        delete that[REG][name];
        cleanup(that);
      } else {
        getAllEventNames(that).forEach(function (name) {
          clear(getHandlers(that, name));
        });
        delete that[REG];
      }
      
      return that;
    }
  };

  function TinyEvents(obj) {
    if (!obj && this instanceof TinyEvents) {
      obj = this;
    }
    if (TinyEvents.isOn(obj)) {
      return obj;
    }
    obj.emit = fns.emit;
    obj.on = obj.addListener = fns.on;
    obj.once = fns.once;
    obj.removeListener = fns.removeListener;
    obj.removeAllListeners = fns.removeAllListeners;
    return obj;
  }

  /**
   * @param obj
   * @returns {boolean} if object has tinyevents on it already
   */
  TinyEvents.isOn = function (obj) {
    return !!(obj && typeof obj === 'object' && obj.emit === fns.emit);
  };

  return TinyEvents;
});
