
/*
 * This Node.js Javascript code demonstrates how to access the One Sight Solutions ossRestApiServer API.
 *  Contact One Sight Solutions for more information.
 *  https://onesight.solutions/
 */

/**
 * Require the 'https' Node.js module.
 * This provides the capabilty to make HTTPS requests.
 */
const https = require( 'https' );

/**
 * Disable the 'NODE_TLS_REJECT_UNAUTHORIZED' option.
 * This prevents errors when HTTPS certificates cannot be authorised.
 * This is a potential security risk, it is the responsibility of the
 *  user to understand the correct, secure use of TLS.
 * This option is included simply for the ease of use of this example.
 */
process.env[ 'NODE_TLS_REJECT_UNAUTHORIZED' ] = 0;

/**
 * Define the ossRestApiServer host name.
 */
const apiServerHost = 'localhost';

/**
 * Define the Client Credentials
 * Note that if you are hardcoding these strings then
 *  both back-slashes (\) and single quotes (') need to be escaped with a back-slash.
 */
const clientId = 'REPLACE-WITH-ACTUAL-APPLICATION-CLIENT-ID';
const clientSecret = 'REPLACE-WITH-ACTUAL-APPLICATION-CLIENT-SECRET';

/**************************************************************************************************
 * Make an API request that lists the values of all points available from the API.
 */
processApiRequest( listAllPoints );

/**************************************************************************************************
 * Process an API request.
 * This function first gets an API access token, then passes that token to the supplied function.
 * The provided function can then use the access token to get specific data from the API.
 *
 * @param completionFunction - The function that will be called when a valid token is received
 */
function processApiRequest( completionFunction )
{

    // Take the clientId and clientSecret and encode to base64
    const encodedCredentials = Buffer.from( clientId + ':' + clientSecret ).toString( 'base64' );

    // Define all the OAuth token request parameters
    let tokenOptions =
    {
        hostname: apiServerHost,
        path: '/ossRestApiServer/token',
        method: 'POST',
        headers:
        {
            'Content-Type': 'application/x-www-form-urlencoded',
            'Authorization': 'Basic ' + encodedCredentials
        }
    };

    // Create the API token request and define the callback functions
    let tokenRequest = https.request( tokenOptions, function( tokenResponse )
    {

        // Reset the accumulated response data
        let responseData = '';

        // Define the callback function that handles received data
        tokenResponse.on( 'data', function( responseChunk )
        {
            // Append the received chunk of data to the accumulated response data
            responseData += responseChunk;
        });

        // Define the callback function that is called when the request ends
        tokenResponse.on( 'end', function()
        {
            // Check the HTTP status code, 200 is OK, the request has been successful
            if( tokenResponse.statusCode == 200 )
            {
                // The request was successful so the response will be JSON data
                // Parse the returned JSON text to native javascript data structures
                let tokenData = JSON.parse( responseData );

                // Extract the access token from the returned data
                let accessToken = tokenData.access_token;

                console.log( 'Received token: ' + accessToken );

                if( completionFunction )
                {
                    // If the supplied completion function is defined
                    //  then call the function, passing it the access token.
                    completionFunction( accessToken );
                }

            }
            else
            {
                // Request failed
                console.error( 'API Request error, status = ' + tokenResponse.statusCode );
                console.error( 'API Request error, response =\n' + responseData );
            }
        });

    });

    // Define the callback function that handles communication errors
    tokenRequest.on( 'error', function( requestError )
    {
        console.error( 'API Request error = ' + requestError );
        console.error( requestError );
    });

    // Send the OAuth token request body data
    tokenRequest.write( 'grant_type=client_credentials&scope=read' );

    // Complete the API token request
    tokenRequest.end();

}

/**************************************************************************************************
 * List the values of all points available from the API.
 *
 * @param accessToken - The API access token
 */
function listAllPoints( accessToken )
{

    // Define all the API list points request parameters
    let tokenOptions =
    {
        hostname: apiServerHost,
        path: '/ossRestApiServer/query/listPoints',
        method: 'GET',
        headers:
        {
            'Authorization': 'Bearer ' + accessToken
        }
    };

    // Create the API list points request and define the callback functions
    let tokenRequest = https.request( tokenOptions, function( tokenResponse )
    {

        // Reset the accumulated response data
        let responseData = '';

        // Define the callback function that handles received data
        tokenResponse.on( 'data', function( responseChunk )
        {
            // Append the received chunk of data to the accumulated response data
            responseData += responseChunk;
        });

        // Define the callback function that is called when the request ends
        tokenResponse.on( 'end', function()
        {
            // Check the HTTP status code, 200 is OK, the request has been successful
            if( tokenResponse.statusCode == 200 )
            {
                // The request was successful so the response will be JSON data.
                // Parse the returned JSON text to native javascript data structures.
                // The returned data will be an array of slot path strings.
                let pointListData = JSON.parse( responseData );

                console.log( 'Found ' + pointListData.length + ' points' );

                // Loop through all points in the returned points list
                for( let slotPath of pointListData )
                {
                    // Read and show the value of each point
                    showSinglePoint( accessToken, slotPath );
                }

            }
            else
            {
                // Request failed
                console.error( 'API Request error, status = ' + tokenResponse.statusCode );
                console.error( 'API Request error, response =\n' + responseData );
            }
        });

    });

    // Define the callback function that handles communication errors
    tokenRequest.on( 'error', function( requestError )
    {
        console.error( 'API Request error = ' + requestError );
        console.error( requestError );
    });

    // Complete the API list points request
    tokenRequest.end();

}

/**************************************************************************************************
 * Show the slot path and value of the specified point.
 *
 * @param accessToken - The API access token
 * @param pointPath - The point path string for the point to show
 */
function showSinglePoint( accessToken, pointPath )
{

    // Define all the API read point value request parameters
    let tokenOptions =
    {
        hostname: apiServerHost,
        path: '/ossRestApiServer/query/readPoint' + pointPath,
        method: 'GET',
        headers:
        {
            'Authorization': 'Bearer ' + accessToken
        }
    };

    // Create the API read point value request and define the callback functions
    let tokenRequest = https.request( tokenOptions, function( tokenResponse )
    {

        // Reset the accumulated response data
        let responseData = '';

        // Define the callback function that handles received data
        tokenResponse.on( 'data', function( responseChunk )
        {
            // Append the received chunk of data to the accumulated response data
            responseData += responseChunk;
        });

        // Define the callback function that is called when the request ends
        tokenResponse.on( 'end', function()
        {
            // Check the HTTP status code, 200 is OK, the request has been successful
            if( tokenResponse.statusCode == 200 )
            {
                // The request was successful so the response will be JSON data.
                // Parse the returned JSON text to native javascript data structures.
                // The returned data will be an single point object with a value key.
                let pointData = JSON.parse( responseData );

                console.log( 'Point data: ' + pointPath + ' = ' + pointData.value );

            }
            else
            {
                // Request failed
                console.error( 'API Request error, status = ' + tokenResponse.statusCode );
                console.error( 'API Request error, response =\n' + responseData );
            }
        });

    });

    // Define the callback function that handles communication errors
    tokenRequest.on( 'error', function( requestError )
    {
        console.error( 'API Request error = ' + requestError );
        console.error( requestError );
    });

    // Complete the API read point value request
    tokenRequest.end();

}

/**************************************************************************************************
 * End of File
 *************************************************************************************************/
